#include "rcomplex.hpp"
#include "frame.hpp"
#include "cursor.hpp"
#include "button.hpp"
#include "stdlib.h" //for random(int)

#define ABS(x) (((x)>0)?(x):(-x)) //Macro is type independent

#define START_FLOCKCOUNT 20
#define MAX_FLOCKCOUNT 200
/* BIG_HASHLENGTH should be about 9 times as big as SMALL_HASHLENGTH, and
both should be prime.  The hasharrays are made of far pointers, at four
bytes per entry, so if BIG_HASHLENGTH is over 16K, then hasharray has to
be a huge pointer.  Let's stay away from that and let BIG_HASHLENGTH be
the largest prime less than 16K, and let SMALL_HASHLENGTH be the largest
prime less than 16K/9.  Using Mathematica, I find that these are
15991 and 1777.  (Note that 9*1777 is 15993.)*/
#define BIG_HASHLENGTH 15991
#define SMALL_HASHLENGTH 1777
#define NON_HIGHLIGHT_BOX 2 //dull green
#define HIGHLIGHT_BOX 15 //white

//Use SQRT2 in JuliaArray.Normalize
#define SQRT2 1.41
//The next numbers are for use by Julia.Randomize
#define RANGE 1.1

class Vpixel //6 bytes
{
private:
	int pixelx, pixely;
	unsigned int hitcount;
	friend class Imagepair;
public:
	Vpixel():pixelx(0),pixely(0),hitcount(0){}
	unsigned char operator==(Vpixel &); //TRUE(1) if xs,ys equal
	unsigned char operator<(Vpixel &);//TRUE if y1<y2 or y1==y2,x1<x2
	friend int Imagepair_sort(const void *r1, const void *r2);
	/* Borland qsort needs this function to sort and array of
	Imagepairs.  We don't make this a member function of Imagepair
	because C++ is somewhat hostile to function pointers.  But it
	has to be a friend of Vpixel so we can look at the Vpixel fields.*/
};


class Imagepair //17 bytes
{
	Complex root;
	Vpixel vpixel;
	unsigned char inframe;
	int color;
	friend class Julia; //for  ShowCopy
public:
	Imagepair():root(0.0,0.0),vpixel(),inframe(0),color(0){}
	void Stow(Complex& point, Frame * fp, Vpixel **ha, unsigned long hl);
	/* This copies point to root and then calls the Frame's
		PixelToReal method on root to fill the vpixel's pixelx and
		pixely fields and to fill the inframe field.  Then it stows a
		copy of vpixel, relative to the base frame, in the hasharray
		if it's offscreen.  If there is a match, then Stow bumps the
		hitcount of the prestowed Vpixel. Else Stow inserts stowpixel
		with a hitcount of 1. And the Imagepair's vpixel's hitcount is
		set to the same as the stowed value.
		HitcountToColor is used to compute the color based on the
		hitcount.*/
	void HitcountToColor(){(vpixel.hitcount>15)?color=15:color=vpixel.hitcount;}
	/* We can use other methods here if we like. For instance
		{color=(vpixel.hitcount%16)+1;} would be okay.*/
	void Show(Frame *fp){if(inframe) fp->pixel(vpixel.pixelx, vpixel.pixely, color);}
	friend int Imagepair_sort(const void *r1, const void *r2);
	/* Borland qsort needs this function to sort and array of
	Imagepairs.  We don't make this a member function of Imagepair
	because C++ is somewhat hostile to function pointers.  But it
	has to be a friend so we can look at the Imagepair fields.*/
};

class Julia
{
	Frame *frameptr;
	int degree; //We'll look at degree 2 and 3, and maybe 4
	Complex c,k,q; // c for degree 2, c&k for 3, c&k&q for 4.
	Complex *image;
	/* This is just a little helper array, 2 or 3 or 4 long, for
		using in the coputations.*/
	int flockcount;
	/* The length of flock is flockcount. Flockimage has length
		degree*flockcount.*/
	Complex *flock;
	Imagepair *flockimage;
	/* Size of this is 17 * degree * flockcount.  If degree is
		4 and flockcount 100, we have 17*4*100 = 6800 < 7K, no prob.
		We would need a huge instead of far pointer if flockcount
		is bigger than 64000/(17*4), about 1000.  So that's why
		we defined MAX_FLOCKCOUNT to be 800 at the head of this file.*/
	Vpixel **hasharray;  /* An array of pointers to VPixels.*/
	unsigned long hashlength; /* This is BIG_HASHLENGTH or
		SMALL_HASHLENGTH.  I want it to be unsigned long, so I can
		be comfortable hashing a pixelx, pixely pair into 
		(pixelx*pixely)%hashlength without an overflow, given that
		(pixelx*pixely) might be about 300,000.*/
	friend class JuliaArray;
public:
	Julia();
	void InstallDisplay(Display *dp);
	~Julia();
	void Free();
	void Reset(unsigned char oneflag);
	void Randomize();
	void Copyparams(Julia &j); //copy c,k,q
	void Bump_c(Complex &dc){c += dc;}
	void Bump_k(Complex &dk){k += dk;}
	void Bump_q(Complex &dq){q += dq;}
	void Mult_flockcount(Real mfc, Real max);
	int Get_flockcount(){ return flockcount;}
	void Zoom(Real mzs){frameptr->mult_zoomscale(mzs);}
	void Pan(Real dx, Real dy){frameptr->pan(dx,dy);}
	void Set_degree(int deg){degree = deg;}
	int Get_degree(){return degree;}
	void JuliaFixed();
	void JuliaPreimage(Complex &z);
	void JuliaFixed2();
	void JuliaPreimage2(Complex &z);
	void JuliaFixed3();
	void JuliaPreimage3(Complex &z);
	void JuliaFixed4(){QuarticSolve(0.0, q, k-1.0, c, image);}
	/* This places into image the four complex solutions to the
	equation z4 + qz2 + kz + c = z, or z4 + qz2 + (k-1)z + c = 0.*/
	void JuliaPreimage4(Complex &w){QuarticSolve(0.0, q, k, c-w, image);}
	/* This places into image the four complex solutions to the
	equation z4 + qz2 + kz + c = w, or z4 + qz2 + kz + c - w = 0 */
	void ComputeStow();
	/* The Compute part of this goes down the flock array, and fills the
		root fields of the Imagepair fields in the flockimage array.
		The Stow part of this takes each Imagepair whose root field
		is already set, and then fixes its vpixel field by compting
		the vpixel's coords, putting it in the hitlist, and getting
		the hitcount, and then puts the Imagepair into the flockimage
		array.*/
	void Sort();
	/* This uses Imagepair_sort and Borland qsort to sort the
		flockimage.*/
	void ShowCopy();
	/* This calls the VPixel Show() method on the first part of the
		flockimage, and also copies the root fields over to the
		flock.  Keep in mind that flockimage will have degree times
		as many members as flock.  We do the Show and Copy in one
		call so that we only have to walk the array once.*/
	void Update();
	/* This calls ComputeStow all down the line of the
		flockimage, calls Sort on flockimage, calls Show on the
		low values of the Imagepairs, and then copies the root fields
		of the low Imagepairs into the flock array*/
	friend ostream& operator<<(ostream &s, Julia &j);
	friend istream& operator>>(istream &s, Julia &j);
};


class JuliaArray
{
	int size;
	unsigned char oneflag; //TRUE if you are displaying only one
	int focusindex;
	Julia *arr;
	int xcount, ycount;
	Display *disp;
	Real step;
	char ext[5];  //These two are for the Load and Save methods.
	char filename[20];
	ButtonBox *Buttons;
	Cursor *cur;
public:
	JuliaArray(int nucount);
	void InstallDisplay(Display *dp);
	void Reset();
	void Locate(int xsize, int ysize);
	int Update();

/* jularray owns a Real step variable that specifies how big
the Bump_? steps should be.  The arguments to these calls simply
specify which of the re and im fields of the Complex to bump, and
in which direction, so the arguments are normall -1.0, 0.0, or 1.0*/
	void Bump_c(Real dx, Real dy);
	void Bump_k(Real dx, Real dy);
	void Bump_q(Real dx, Real dy);
	void Mult_flockcount(Real mfc);
	void Zoom(Real mzs);
	void Pan(Real dx, Real dy);
	void Set_degree(int deg);
	int Get_degree(){ return arr[0].Get_degree();}
	void Bump_focus();
	void Set_count(int nucount);
	void Use_focus();
	void Normalize();
	void Draw_boxes();
	void Mult_step(Real mult){step *= mult;}
	void Randomize();
	unsigned char Pickaname(char far *title);  //Code in files.cpp
	void Load();
	void Save();
	void Update_focus(Cursor *cur);
	void Set_buttons(ButtonBox *iButtons){ Buttons = iButtons;}
	void Set_cursor(Cursor *icur){ cur = icur;}
	int Get_oneflag(){return oneflag;}
	int Hit(Cursor cur);
};

