#include <fstream.h> //for load and save
#include "julia.hpp"
#include "memcheck.hpp"
#include "cursor.hpp"
#include "keycode.hpp"
#include "math.h"
#define NORM 50

//These 3 belong in a different file for random.cpp-------------------

Real randomreal()
{
	return (random(10000)/9999.0); //Random real from 0.0 to 1.0
}

Real randomsignreal()
{
	return (-1.0+2.0*randomreal()); //Random real from -1.0 to 1.0
}

Complex randomcomplex()
{
	Real arg = randomreal()*2.0*PI;
	return randomreal()*Complex(cos(arg), sin(arg));
	//Complex with magnitude 0 to 1 and random argument.
}

Complex randomunitcomplex()
{
	Real arg = randomreal()*2.0*PI;
	return Complex(cos(arg), sin(arg));
	//Complex with unit magnitude and random argument.
}
//------------------end Random.cpp

inline unsigned char Vpixel::operator==(Vpixel &vp)
{
	if (vp.pixelx == pixelx && vp.pixely == pixely)
		return TRUE;
	return FALSE;
}

inline unsigned char Vpixel::operator<(Vpixel &vp)
{
	if (pixely < vp.pixely)
		return TRUE;
	if (pixely == vp.pixely && pixelx < vp.pixelx)
		return TRUE;
	return FALSE;
}

void Imagepair::Stow(Complex &point, Frame *fp, Vpixel **hasharr,
	unsigned long hashlen)
{
	unsigned long baseindex, hashindex;

	root = point; //Use Complex::operator=
	inframe = fp->RealToPixel(root.get_re(), root.get_im(), vpixel.pixelx,
		vpixel.pixely);
	if (inframe == TRUE)
		vpixel.hitcount = (fp->readpixel(vpixel.pixelx, vpixel.pixely))+1;
	else
	{
/*---Begin to stow a copy of the pixel in vpl.  But first, convert it to
	base, unzoomed coordinates so there won't be so many cases. */
		Vpixel *stowpixel = new Vpixel;
#define BASEFRAME
/* If you don't use the baseframe, then after a few zooms in, there
are so many offscreen pixels that you get the "Hashtable is full" 
message.  But using the baseframe cuts down on the accuracy.  You might
use a frame that's somewhat zoomed in as the baseframe, maybe to two
levels.*/
#ifdef BASEFRAME
		fp->RealToBasePixel(root.get_re(), root.get_im(), stowpixel->pixelx,
			stowpixel->pixely); //To keep down the number of cases.
#else //not BASEFRAME
	stowpixel->pixelx = vpixel.pixelx;
	stowpixel->pixely = vpixel.pixely;
#endif //BASEFRAME
/*Here comes the hash function.  Multiply the pixel values and
take the result modulo hashlen.  We multiply rather than adding so as
to spread the values out more.  If we added, we'd get a max sum of about
480+640, which is much less than BIG_HASHLENGTH of about 16K.*/
		baseindex = hashindex = ABS( (long)(stowpixel->pixelx) *
			(long)(stowpixel->pixely) ) % hashlen;
		//Find the first match or the first blank.
		int found = 0;
		//0 means not found, 1 means found a match, 2 means found a NULL
		while (!found)
		{
			if (hasharr[hashindex] == NULL)
			{
				found = 2;
				break;
			}
			if (*(hasharr[hashindex]) == *stowpixel)
			{
				found = 1;
				break;
			}
			if (++hashindex >= hashlen)
				hashindex = 0;
			if (hashindex == baseindex)
			{
					cout << "Hash table is full!/n";
					cout << "Press any key to exit program./n";
					while (!kbhit()) //Wait for a keypress
						;
					exit(1); //Terminate with extreme prejudice.
			}
		}
		if (found==2)
		{
			hasharr[hashindex] = stowpixel;
			stowpixel->hitcount = 1;
			vpixel.hitcount = 1;
		}
		else //found == 1
		{
			delete stowpixel;
			hasharr[hashindex]->hitcount++;
			if (hasharr[hashindex]->hitcount > fp->readmaxcolor())
				hasharr[hashindex]->hitcount =  fp->readmaxcolor();
			vpixel.hitcount = hasharr[hashindex]->hitcount;
		}
	}
	HitcountToColor(); 
}

int Imagepair_sort(const void *r1, const void *r2)
{
	if (((Imagepair *)r1)->vpixel.hitcount <
		((Imagepair *)r2)->vpixel.hitcount)
		return -1;
	if (((Imagepair *)r1)->vpixel.hitcount >
		((Imagepair *)r2)->vpixel.hitcount)
		return 1;
	return 0; //otherwise
}

Julia::Julia()
{
	degree = 4;
	c = Complex(-0.054033, 0.22306034);
	k = Complex(-0.00555353, -0.644392);
	q = Complex(-0.7, 0.4);
	flockcount = START_FLOCKCOUNT;
	flock = NULL;
	flockimage = NULL;
	image = NULL;
	frameptr = NULL;
	hasharray = NULL;
	hashlength = SMALL_HASHLENGTH;
}

void Julia::InstallDisplay(Display *dp)
{
	frameptr = new Frame(dp);
}

Julia::~Julia()
{
	if (flock)
		delete flock;
	if (flockimage)
		delete flockimage;
	if (image)
		delete image;
	if (hasharray)
	{
		for (unsigned long i=0; i<hashlength; i++)
			if (hasharray[i])
				delete hasharray[i];
		delete hasharray;
	}
}

void Julia::Free()
{
	if (flock)
	{
		delete flock;
		flock = NULL;
	}
	if (flockimage)
	{
		delete flockimage;
		flockimage = NULL;
	}
	if (image)
	{
		delete image;
		image = NULL;
	}
	if (hasharray)
	{
		for (unsigned long i=0; i<hashlength; i++)
		{
			if (hasharray[i])
				delete hasharray[i];
		}
		delete hasharray;
		hasharray = NULL;
	}
}

void Julia::Reset(unsigned char oneflag)
{
	Real framerange = 0.0;

	/* Don't call Free() because it already will have been called by
		JuliaArray::Reset()*/
	flock = new Complex[flockcount];
	newcheck(flock, "Julia::Reset");
	flockimage = new Imagepair[degree * flockcount];
	newcheck(flockimage, "Julia::Reset");
	image = new Complex[degree];
	newcheck(image, "Julia::Julia");
	if (oneflag)
		hashlength = BIG_HASHLENGTH;
	else
		hashlength = SMALL_HASHLENGTH;
	hasharray = new Vpixel*[hashlength];
	/* Note: "new (Vpixel*)[HASHCOUNT]" won't compile!  Moral: You must
		use NO PARENS around the type of array members in a "new".*/
	newcheck(hasharray, "Julia::Reset");
	for (unsigned long i=0; i<hashlength; i++)
		hasharray[i] = NULL;
	//Put the fixed points in the image array.
	JuliaFixed();
	//Start with the fixed point with the largest magnitude.
	flock[0] = image[0];
	framerange = flock[0].Magnitude();
	for (int j=1; j<degree; j++)
		if (image[j].Magnitude() > framerange)
		{
			flock[0] = image[j];
			framerange = flock[0].Magnitude();
		}
	//Now fill flock with flockcount random iterates while checking framerange.
	for (j = 1; j<flockcount; j++)
	{
		JuliaPreimage(flock[j-1]);
		flock[j] = image[random(degree)];
		if (flock[j].Magnitude() > framerange)
			framerange = flock[j].Magnitude();
	}
	frameptr->setrange(framerange);
 //	frameptr->clear();   /* let JuliaArray do this*/
}

void Julia::Copyparams(Julia &j)
{
	c = j.c;
	k = j.k;
	q = j.q;
}

void Julia::Mult_flockcount(Real mfc, Real max)
{
	flockcount *= mfc;
	if (flockcount<degree)
		flockcount=degree;
	if (flockcount>max)
		flockcount = max;
//	Reset(); //Let jularray do it.
}

void Julia::ComputeStow()
{
	unsigned int j;

	for (unsigned int i=0; i < flockcount; i++)
	{
		JuliaPreimage(flock[i]);
		for (j=0; j<degree; j++)
			flockimage[(degree * i) + j].Stow(image[j],
			frameptr, hasharray, hashlength);
	}
}

void Julia::Sort()
{
	qsort(flockimage, degree*flockcount, sizeof(Imagepair), Imagepair_sort);
}

void Julia::ShowCopy()
{
	for (unsigned int i=0; i<flockcount; i++)
	{
		flockimage[i].Show(frameptr);
		flock[i] = flockimage[i].root;
	}
}

void Julia::Update()
{
	ComputeStow();
	Sort();
	ShowCopy();
}

void Julia::Randomize()
{
	c = k = q = RANGE*Complex(1.0, 1.0);
	while (c.Magnitude() + k.Magnitude() + q.Magnitude() >
		sqrt(degree)*RANGE)
	{
		c = RANGE*randomcomplex();
		k = RANGE*randomcomplex();
		q = RANGE*randomcomplex();
	}
}

//--------------------------
void Julia::JuliaFixed()
{
	switch (degree)
	{
		case 2:
			JuliaFixed2();
			break;
		case 3:
			JuliaFixed3();
			break;
		case 4:
			JuliaFixed4();
			break;
	}
}

void Julia::JuliaPreimage(Complex &z)
{
	switch (degree)
	{
		case 2:
			JuliaPreimage2(z);
			break;
		case 3:
			JuliaPreimage3(z);
			break;
		case 4:
			JuliaPreimage4(z);
			break;
	}
}

void Julia::JuliaFixed2()
{
/* A fixed point for the equation z = z^2 + c.  Use
quadratic formula on z^2 - z + c to get z equal
0.5 plus or minus 0.5 * sqrt (1.0 - 4.0 * c).*/
	Complex temp;

	temp = 0.5 * Sqrt(1.0 - (4.0 * c));
	image[0] = 0.5 - temp;
	image[1] = 0.5 + temp;
}

void Julia::JuliaPreimage2(Complex& z)
{
/* Find the solution to the equation z = pre^2 + c. This
means pre is plus or minus Sqrt (z - c). */
	Complex temp;

	temp = z - c;
	image[0] = Sqrt(temp);
	image[1] = -(Sqrt(temp));
}

void Julia::JuliaFixed3()
{
#define NEWROOT
#ifndef NEWROOT
/* Find the fixed points to the equation z = z^3 + kz + c,
	means solve  0 = z^3 + (k-1)z + c, for z.*/

	Complex p, nuk, cand[6];
	int index1 = 0, index2;

	p = 0.5 * c;
	nuk = k - Complex(1.0,0);
	Cuberoot(p + Sqrt( Cube(nuk/3.0) + Square(p) ),
		image[0], cand[1], cand[2]);
	Cuberoot(p - Sqrt( Cube(nuk/3.0) + Square(p) ),
		cand[3], cand[4], cand[5]);
	for (int i=1; i<6; i++)
	{
		if (!index1 && !(cand[i] == image[0]))
			index1 = i;
		if (index1 && (!(cand[i] == image[0])) &&
			(!(cand[i] == cand[index1])) )
		{
			index2 = i;
			break;
		}
	}
	image[0] = image[0] - (nuk / (3.0 * image[0]));
	image[1] = cand[index1] - (nuk / (3.0 * cand[index1]));
	image[2] = cand[index2]- (nuk / (3.0 * cand[index2]));
#else
	CubicSolveLeadZero(0.0, k-1, c, image);
#endif //NEWROOT
}

void Julia::JuliaPreimage3(Complex &w)
{
/* Find the solution to the equation w = z^3 + kz + c, for z,
	means solve 0 = z^3 + kz + (c-w), for z.*/

#ifndef NEWROOT

	Complex p, cand[6];
	int index1 = 0, index2;

	p = 0.5 * (w - c);
	Cuberoot( p + Sqrt( Cube(k/3.0) + Square(p) ),
		image[0], cand[1], cand[2]);
	Cuberoot( p - Sqrt( Cube(k/3.0) + Square(p) ),
		cand[3], cand[4], cand[5]);
	for (int i=1; i<6; i++)
	{
		if (!index1 && !(cand[i] == image[0]))
			index1 = i;
		if (index1 && (!(cand[i] == image[0])) &&
			(!(cand[i] == cand[index1])) )
		{
			index2 = i;
			break;
		}
	}
	image[0] = image[0] - (k / (3.0 * image[0]));
	image[1] = cand[index1] - (k / (3.0 * cand[index1]));
	image[2] = cand[index2]- (k / (3.0 * cand[index2]));
#else
	CubicSolveLeadZero(0.0, k, c-w, image);
#endif //NEWROOT
}
//------JuliaArray Methods

JuliaArray::JuliaArray(int nucount)
{
	size = nucount;
	oneflag = FALSE;
	focusindex = 4;
	arr = new Julia[size];
	newcheck(arr, "JuliaArray::JuliaArray");
	xcount = ycount = (int)sqrt(size);
	step = 0.1;
	strcpy(ext,".JUL");
}

void JuliaArray::InstallDisplay(Display *dp)
{
	disp = dp;
	for (int i=0; i< size; i++)
		arr[i].InstallDisplay(dp);
}

void JuliaArray::Reset()
{
	Buttons->ValButtonArray[DEGREE]->Set_intval(arr[focusindex].Get_degree());
	Buttons->ValButtonArray[FLOCKCOUNT]->Set_intval(arr[focusindex].Get_flockcount());
	Buttons->ValButtonArray[STEPSIZE]->Set_realval(step);
	for (int i=0; i< size; i++)
		arr[i].Free();
	if (oneflag)
		arr[focusindex].Reset(1);
	else
		for (i=0; i< size; i++)
			arr[i].Reset(0);
	disp->clear();
	if(cur->Getx() < 100)
	  cur->Hidecursor();
	Draw_boxes();
	Buttons->Show();
	if(cur->Getx() < 100)
	  cur->Showcursor();
}

int JuliaArray::Update()
{  int cursorcode,hitval= -1,j;
	if (oneflag)
	  {
		 arr[focusindex].Update();
		 if(cur->MouseMoved())
			 Update_focus(cur);
		 cursorcode = cur->Update(0);
		 if ((cursorcode == LEFT_BUTTON) || (cursorcode == RIGHT_BUTTON))
		 {
			hitval = Buttons->Hit(cursorcode,cur);
			if((hitval < 0) && (cursorcode == RIGHT_BUTTON) && (Hit(*cur)))
			  cur->Moveto(50,65); // if right-click on julia side
		 }
	  }
	else
	{
	  for (int i=0; i< size; i++)
	  {
			arr[i].Update();
			if(cur->MouseMoved())
			Update_focus(cur);
			cursorcode = cur->Update(0);
			if ((cursorcode == LEFT_BUTTON) || (cursorcode == RIGHT_BUTTON))
			{
				hitval = Buttons->Hit(cursorcode,cur);
				if(hitval >= 0)
				  return hitval;
				if((cursorcode == LEFT_BUTTON) && (Hit(*cur)))
				  return NORM; // if left-click on julia side...Normalize
				else if((cursorcode == RIGHT_BUTTON) && (cur->Getx()>= 100))
				cur->Moveto(50,65); // if right-click on julia side...Move
			}
		}
	}
	return hitval;
}

#define FRAMESPACE 2

void JuliaArray::Locate(int xsize, int ysize)
{
	int width;
	int height;
	int row, column;

	if (xsize*ysize > size)
		return;
	xcount = xsize;
	ycount = ysize;
	if (xcount == 1 && ycount == 1)
		oneflag = TRUE;
	else
		oneflag = FALSE;
/* Now need to figure out min and max values for a frame for each
of the arr[i].frameptr frames.*/
	width = (disp->readmaxx() -
		disp->readminx() ) / xsize;
	width -= 2*FRAMESPACE;
	height = (disp->readmaxy() -
		disp->readminy() ) / ysize;
	height -= 2*FRAMESPACE;
	if (!oneflag)
		for (int i=0; i<size; i++)
		{
			row = i / ycount;
			column = i - (row * xcount);
			int left, top, right, bottom;
			left = FRAMESPACE + column*(width + 2*FRAMESPACE) + disp->readminx();
			right = left + width;
			top = FRAMESPACE + row*(height + 2*FRAMESPACE);
			bottom = top + height;
			arr[i].frameptr->Set(left,top,right,bottom);
		}
	else
		arr[focusindex].frameptr->Set(FRAMESPACE + disp->readminx(),
				 FRAMESPACE, width + disp->readminx(), height);
};

void JuliaArray::Bump_c(Real dx, Real dy)
{
	Complex dc(dx,dy);

	dc *= step;
	if (oneflag)
		arr[focusindex].Bump_c(dc);
	else
		for (int i=0; i<size; i++)
			arr[i].Bump_c(dc);
	cur->Hidecursor();
	Buttons->ShowCKQButtons(arr[focusindex].c.get_re(),arr[focusindex].
	  c.get_im(),arr[focusindex].k.get_re(),arr[focusindex].k.get_im(),
	  arr[focusindex].q.get_re(),arr[focusindex].q.get_im());
	cur->Showcursor();
	Reset();
}

void JuliaArray::Bump_k(Real dx, Real dy)
{
	Complex dc(dx,dy);

	dc *= step;
	if (oneflag)
		arr[focusindex].Bump_k(dc);
	else
		for (int i=0; i<size; i++)
			arr[i].Bump_k(dc);
	cur->Hidecursor();
	Buttons->ShowCKQButtons(arr[focusindex].c.get_re(),arr[focusindex].
	  c.get_im(),arr[focusindex].k.get_re(),arr[focusindex].k.get_im(),
	  arr[focusindex].q.get_re(),arr[focusindex].q.get_im());
	cur->Showcursor();
	Reset();
}

void JuliaArray::Bump_q(Real dx, Real dy)
{
	Complex dc(dx,dy);

	dc *= step;
	if (oneflag)
		arr[focusindex].Bump_q(dc);
	else
		for (int i=0; i<size; i++)
			arr[i].Bump_q(dc);
	cur->Hidecursor();
	Buttons->ShowCKQButtons(arr[focusindex].c.get_re(),arr[focusindex].
	  c.get_im(),arr[focusindex].k.get_re(),arr[focusindex].k.get_im(),
	  arr[focusindex].q.get_re(),arr[focusindex].q.get_im());
	cur->Showcursor();
	Reset();
}

void JuliaArray::Mult_flockcount(Real mfc)
{
	if (oneflag)
		arr[focusindex].Mult_flockcount(mfc, (Real)MAX_FLOCKCOUNT);
	else
		for (int i=0; i<size; i++)
			arr[i].Mult_flockcount(mfc, (Real)MAX_FLOCKCOUNT/2.0);
	//Reset();
}

void JuliaArray::Zoom(Real mzs)
{
	if (oneflag)
		arr[focusindex].Zoom(mzs);
	else
		for (int i=0; i<size; i++)
			arr[i].Zoom(mzs);
	Reset();
}

void JuliaArray::Pan(Real dx, Real dy)
{
	if (oneflag)
		arr[focusindex].Pan(dx,dy);
	else
		for (int i=0; i<size; i++)
			arr[i].Pan(dx,dy);
	Reset();
}

void JuliaArray::Set_degree(int deg)
{
	if((arr[0].Get_degree() == deg) || (deg < 2) || (deg > 4))
	  return;
	/*if (oneflag)
		arr[focusindex].Set_degree(deg);
	else       */
		for (int i=0; i<size; i++)
			arr[i].Set_degree(deg);
	Reset();
}

void JuliaArray::Set_count(int nucount)
{

	if (nucount == xcount * ycount)
		return; //No change.
	if (nucount == 9) //If we're here we've already changed to one_flag.
	{
		Mult_flockcount(1/3.0); /*Change focus back to small flock.
			one_flag is still on here, but about to go off.*/
		Locate(3, 3);
	}
	else
	{
		Locate(1,1);
		Mult_flockcount(3.0); /* Oneflag is on, so this will
			Change flock to big on focus flock*/
	}
	Buttons->Set_oneflag(oneflag);
	Reset();
}

void JuliaArray::Use_focus()
{
	if (oneflag)
		return;
	arr[4].Copyparams(arr[focusindex]); //Use Julia::operator=
	focusindex = 4;
	Normalize();
	Locate(3,3);
	Reset();
	cur->Showcursor();
	cur->Moveto(arr[4].frameptr->GetMidx(),arr[4].frameptr->GetMidy());
	cur->Hidecursor();
}

void JuliaArray::Normalize()
{
	//assume size is 9.
	int i;

	switch (arr[0].degree) //Always assume all have same degree.
	{
		case 2:
			for (i=0; i<9; i++)
				if (i != 4)
					arr[i].c = arr[4].c + step*randomunitcomplex();
			break;
		case 3:
			for (i=0; i<9; i++)
				if (i != 4)
				{
					arr[i].c = arr[4].c +
						sqrt(1.0/2.0)*step*randomunitcomplex();
					arr[i].k = arr[4].k +
						sqrt(1.0/2.0)*step*randomunitcomplex();
				}
			break;
		case 4:
			for (i=0; i<9; i++)
				if (i != 4)
				{
					arr[i].c = arr[4].c +
						sqrt(1.0/3.0)*step*randomunitcomplex();
					arr[i].k = arr[4].k +
						sqrt(1.0/3.0)*step*randomunitcomplex();
					arr[i].q = arr[4].q +
						sqrt(1.0/3.0)*step*randomunitcomplex();
				}
			break;
	}
}

void JuliaArray::Draw_boxes()
{
	if (oneflag)
	{
		arr[focusindex].frameptr->framebox(HIGHLIGHT_BOX); //gray
		return;
	}
	for (int i=0; i<size; i++)
	{
		if (i != focusindex)
			arr[i].frameptr->framebox(NON_HIGHLIGHT_BOX); //green
		else
			arr[focusindex].frameptr->framebox(HIGHLIGHT_BOX); //gray
	}
}

void JuliaArray::Bump_focus()
{
	if (oneflag)
		return; //In size 1, we leave the focus alone.
	arr[focusindex].frameptr->framebox(NON_HIGHLIGHT_BOX); //gray
	if (++focusindex >= size)
		focusindex = 0;
	arr[focusindex].frameptr->framebox(HIGHLIGHT_BOX); //gray
}

void JuliaArray::Randomize()
{
	if (oneflag)
		arr[focusindex].Randomize();
	else
		for (int i=0; i<size; i++)
			arr[i].Randomize();
	Reset();
	cur->Hidecursor();
	Buttons->ShowCKQButtons(arr[focusindex].c.get_re(),arr[focusindex].
	  c.get_im(),arr[focusindex].k.get_re(),arr[focusindex].k.get_im(),
	  arr[focusindex].q.get_re(),arr[focusindex].q.get_im());
	cur->Showcursor();
}

ostream& operator<<(ostream &s, Julia &j)
{
	s << j.degree << "\n";
	s << j.c << "\n";
	s << j.k << "\n";
	s << j.q << "\n";
	return s;
}

istream& operator>>(istream &s, Julia &j)
{
	s >> j.degree;
	s >> j.c;
	s >> j.k;
	s >> j.q;
	j.Reset(0); //assume not oneflag to be safe
	return s;
}

void JuliaArray::Load()
{
	if (Pickaname("LOAD A PICTURE"))
		{
			ifstream ifile;
			ifile.open(filename);
			if (!ifile)
			{
				cout << "Can't open pic file.";
				return;
			}
			ifile >> arr[focusindex];
			ifile.close();
		}
 }

 void JuliaArray::Save()
 {
		if (Pickaname("SAVE THE PICTURE"))
		{
			ofstream ofile;
			ofile.open(filename);
			ofile << arr[focusindex];
			ofile.close();
		}
 }

void JuliaArray::Update_focus(Cursor *cur)
{
	if(cur->Getx()<100)
	 {
	  if(!cur->CursorVisible())
		 cur->Showcursor();
	  return;
	 }
	else
	  if(cur->CursorVisible())
		 cur->Hidecursor();
	if (oneflag)
		return; //In size 1, we leave the focus alone.
	for(int i=0;i<size;i++)
	  {
		 if(((arr[i].frameptr->GetMinx()-4) < cur->Getx()) &&
			(cur->Getx() <= (arr[i].frameptr->GetMaxx()+4)) &&
			((arr[i].frameptr->GetMiny()-4) < cur->Gety()) &&
			(cur->Gety() <= (arr[i].frameptr->GetMaxy())+4))
			{
			  if(focusindex != i) // if not already on focus
				 { arr[focusindex].frameptr->framebox(NON_HIGHLIGHT_BOX); //gray
					focusindex=i;
					arr[focusindex].frameptr->framebox(HIGHLIGHT_BOX);
					Buttons->ShowCKQButtons(arr[focusindex].c.get_re(),arr[focusindex].
					c.get_im(),arr[focusindex].k.get_re(),arr[focusindex].k.get_im(),
					arr[focusindex].q.get_re(),arr[focusindex].q.get_im());
				 }
			  return;
			}
	  }
}

int JuliaArray::Hit(Cursor cur)
{
	if(cur.Getx()<100)
	  return 0;
	if (oneflag)
	{
		if (cur.Getx()>= 100)
			return 1;
		else return 0;
	}
	for(int i=0;i<size;i++)
	  {
		 if(((arr[i].frameptr->GetMinx()-4) < cur.Getx()) &&
			(cur.Getx() <= (arr[i].frameptr->GetMaxx()+4)) &&
			((arr[i].frameptr->GetMiny()-4) < cur.Gety()) &&
			(cur.Gety() <= (arr[i].frameptr->GetMaxy())+4))
			  return 1;
	  }
	return 0;
}
