#ifndef FRAME_HPP
#define FRAME_HPP
#include "display.hpp"
#include "types.h"

class Frame
{
private:
	Display *disp;
	int minx, miny, maxx, maxy;
	int midx, midy, halfx, halfy; 
	/* minx,maxx,miny,maxy define a window inside the correpsonding disp
		values.  (midx, midy) is the display center of the frame window.
		halfx and halfy give the pixel distances of the frame center.
		These values are maintained because they get reused over and
		over in RealToPixel. */
	Real cx, cy;
	
	/* The real number center of the world that you project from.
		Panning in this world changes cx and cy. */
	Real range, scale;
	/* In the Real world, we look at a window ranging from
		-range to +range on the x axis.  RealToPixel uses the
		value scale = midy/range a lot, so we maintain this as a
		private.  Zooming on the Real world changes range (and
		scale).*/
	Real zoomfactor;
		/* This is like a lens aspect of the frame, how much it inherently
		zooms.  So if you set it to a range of r, it will in fact give you
		a range of zoomfactor*r.  This defaults to 1.0.*/

public:
	Frame(){}; //Default constructor just makes everything zero.
	void InstallDisplay(Display *dp); //Puts meaningful values in.
	Frame(Display *dp){InstallDisplay(dp);}
	void Set(int left, int top, int right, int bottom);
	/* If you want multiple frames on a display, you use this
		to adjust the minx, etc. values */
	Real readscale(){return scale;}
	Real readrange(){return range;}
	int GetMinx(){return minx;}
	int GetMiny(){return miny;}
	int GetMaxx(){return maxx;}
	int GetMaxy(){return maxy;}
	int GetMidx(){return midx;}
	int GetMidy(){return midy;}
	void setrange(Real nurange){range = nurange; scale = halfy/(zoomfactor*range);}
	/* scale is really a "personal" variable, so we don't set it
		independently.*/
	void mult_zoomscale(Real mz){zoomfactor *= mz;scale = midy/(zoomfactor*range);}
	void pan(Real dx, Real dy){cx += zoomfactor*dx; cy += zoomfactor*dy;}
	unsigned char RealToPixel(Real rx, Real ry, int &px, int &py);
	/*This returns TRUE if the computed pixel value fits in the
		minx,maxx,miny,maxy box, and returns FALSE otherwise. */
	void RealToBasePixel(Real rx, Real ry, int &px, int &py);
	void pixel(int x, int y, int color){disp->pixel(x,y,color);}
	unsigned char readpixel(int x, int y){return disp->readpixel(x,y);}
	unsigned char readmaxcolor(){return disp->readmaxcolor();}
	void clear(){disp->clear();}
	void framebox(int col) {disp->box(minx-1, miny-1, maxx+1, maxy+1, col);}

};
#endif //FRAME_HPP
