//vine.c   A program to show images found in chapter 14 of
//Clifford Pickover, "Computers, Pattern, Chaos & Beauty".  Program by
//Rudy Rucker, Sept 24, 1990.  Copyright Rudy Rucker (C) 1994.
//This source code is shareware that may be freely copied and shared.
//Any commercial use of this source code must be licensed from Rudy
//Rucker.  Contact: rucker@jupiter.sjsu.edu.
//Build from a project file with
//vine.c, egavga.obj, and cga.obj.  The *.obj files are compiled
//Borland BGI graphics files.
//----------------------------------Includes-----------
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include <graphics.h>
#include <dos.h>
#include <alloc.h>
#include <sys\stat.h>
#include <io.h>
#include <time.h> //For randomize
//---------------------------------Defines-------
#define MAXTRAILLENGTH 2000
#define PI 3.141592653
#define POW(x,y) ((x) > 0) ? pow(x,y) : pow(-(x),y)
#define MIN(x,y) ((x) > (y) ) ? y : x
//---------------------------------Types--------
struct pair
{
	int x;
	int y;
};
struct fpair
{
	double x;
	double y;
};
//------------------------------------Global Variables----
int gdriver, startgdriver, gmode, startgmode, maxx, maxy, maxcolor;
double lox = -20, hix = 20, loy = -16, hiy = 16;
double h = 0.1, alpha = 2.7, beta = 2;
int maxiteration = 50;
int resolution = 10;
int steptype = 4;
/* this decides how to pick the startingpoints.  0 is random startpoints,
	1 is points at a resolution by resolution grid, 2 is all points on a
	diagonal line, 3 is all points on a randomly selected line, 4 is
	points on a viny sine curves. -1 is a triangle curve for speed.*/
long int step = 0;
int screenstep = 0;
int traillength = MAXTRAILLENGTH;
int flavor = 0;
/* this selects which f(x) to use.  0 is sin(x + sin(alpha x)).
  1  is sin(POW(x,beta) + sin(alpha*x)). */
struct fpair memory[MAXTRAILLENGTH];
int showflag = 0, spinflag = 1, exitflag = 0;
int fullflag= 0, eraseflag = 0, textflag = 1;
//------------------------------------Function Declarations--
void startgraphics(void);
void randomizepalette(void);
void spinpalette(void);
struct pair pixel(struct fpair);
void show(struct pair, int);
void trail(void);
double saw(double);
struct fpair randompoint(void);
double randomy(void);
double randomx(void);
struct fpair grid(void);
struct fpair startfp(void);
struct fpair move(struct fpair);
void checkkeyboard(void);
void showparams(void);
void inform(char *);
void readparams(char *);
void writeparams(char *);
char * pickaname(void);
//------------------------------------MAIN--------
int main()
{
	readparams("vine.prm");
	randomize();
//	inform("vine.inf");
//	inform("vine.hlp");
	clrscr();
	printf("\n\n        VINE   (from \"Seek Ye The Gnarl, Pt. I\")\n\n");
printf("This shareware executable and source is copyright (C) Rudy Rucker 1994.\n");
printf("Enjoy it and pass it on.  Any commercial use must be approved by Rudy Rucker,\n");
printf("rucker@jupiter.sjsu.edu\n\n");
printf("While running, press the H key for help, and SHIFT+H for more help.\n\n");
printf("PRESS ENTER TO BEGIN!\n\n");
	while (!kbhit())
		;
	getch();
	startgraphics();
	textflag = 0;
	while (!exitflag)
	{
		trail();
		checkkeyboard();
	}
	restorecrtmode();
	textflag = 1;
//	free( (void *) memory );
	return 0;
}
//-------------------------------------Function Definitions---
void startgraphics(void)
{
	static int firsttime = 1;

	if (firsttime) {
		registerbgidriver( CGA_driver );
		registerbgidriver( EGAVGA_driver );
		detectgraph( &gdriver, &gmode );
		startgmode = gmode;
		startgdriver = gdriver;
		if (gdriver < 0) {
			printf("No graphics hardware detected.\n");
			exit(1);
		}
		if (gmode == CGAHI)
			gmode = CGAC0;
		initgraph( &gdriver, &gmode, "" );
		firsttime = 0;
	}
	else
		initgraph( &gdriver, &gmode, "" );
	maxx = getmaxx();
	maxy = getmaxy();
	maxcolor = getmaxcolor();
} 

void randomizepalette(void)
{
// getpalette type to find numbers VGA is using.  setrgb uses bits
// 2 thru 8, so OR with 12 to turn low two on, to keep bright. 
	static struct palettetype pal;
	int i;
	static int firsttime=1;

	if (gdriver == VGA)
	{
		if (firsttime)
		{
			getpalette(&pal);
			firsttime = 0;
		}
		for (i=1; i<16; i++)
			setrgbpalette( pal.colors[i], random(256) | 12,
				random(256) | 12, random(256) | 12 );
	}
	else if (gdriver == CGA)
	{
		gmode++;
		if (gmode > 4)
			gmode = 0;
		startgraphics();
		textflag = 0;
	}
	else // when gdriver is EGA
		for (i=1; i<16; i++)
			setpalette( i, random(64) );
}

void spinpalette(void)
{
// getpalette type to find numbers VGA is using.  setrgb uses bits
// 2 thru 8, so OR with 12 to turn low two on, to keep bright. 
	static struct palettetype pal;
	static int egapalette[16];
	int base, i;
	static int firsttime=1;


	if (gdriver == VGA)
	{
		if (firsttime)
		{
			getpalette(&pal);
			firsttime = 0;
		}
		else
		{
			base = pal.colors[1];
			for (i=1; i<15; i++)
				pal.colors[i] = pal.colors[i+1];
			pal.colors[15] = base;
			setrgbpalette( pal.colors[15], random(256) | 12,
					random(256) | 12, random(256) | 12 );
		}
	}
	else if (gdriver == EGA)
	{
		if (firsttime)
		{
			for (i=1; i<16; i++)
				egapalette[i] = 1 + random(63);
			firsttime = 0;
		}
		else
		{
				for (i=1; i<15; i++)
				{
					egapalette[i] = egapalette[i+1];
					egapalette[15] = 1 + random(63);
				}
				for (i=1; i<16; i++)
					setpalette( i, egapalette[i] );
		}
	}
}

struct pair pixel(struct fpair fp)
{
	struct pair pix;

	pix.x = (fp.x - lox) * maxx / ( hix - lox );
	pix.y = (hiy - fp.y) * maxy / ( hiy - loy );
	return pix;
}

void show(struct pair p, int color)
{
	putpixel(p.x, p.y, color);
}

struct fpair randompoint(void)
{
	struct fpair fp;

	fp.x = lox + (hix - lox) * (double) random(0x7FFF) / (double) 0x7FFF;
	fp.y = loy + (hiy - loy) * (double) random(0x7FFF) / (double) 0x7FFF;
	return fp;
}

double randomy(void)
{
	return ( loy + (hiy - loy) * (double) random(0x7FFF) /
		(double) 0x7FFF );
}

double randomx(void)
{
	return ( lox + (hix - lox) * (double) random(0x7FFF) /
		(double) 0x7FFF );
}

struct fpair grid(void)
{
	struct fpair fp;

	fp.y = loy + ( ( hiy - loy ) / resolution ) * ( step / resolution );
	fp.x = lox + ( ( hix - lox ) / resolution ) * ( step % resolution );
	return fp;
}

struct fpair move(struct fpair fp)
{
	struct fpair newfp;
	int i;

	if (flavor == -1)
	{
		newfp.x = fp.x - h * saw( fp.y + saw( alpha * fp.y ) );
		newfp.y = fp.y + h * saw( fp.x + saw( alpha * fp.x ) );
	}
	else if (flavor == 0)
	{
		newfp.x = fp.x - h * sin( fp.y + sin( alpha * fp.y ) );
		newfp.y = fp.y + h * sin( fp.x + sin( alpha * fp.x ) );
	}
	else if (flavor == 1)
	{
		newfp.x = fp.x - h * sin( POW( fp.y, beta ) +
			 sin( alpha * fp.y ) );
		newfp.y = fp.y + h * sin( POW( fp.x, beta ) +
			 sin( alpha * fp.x ) );
	}
	else if (flavor == 2)
	{
		newfp.x = fp.x - h * sin( fp.y + sin(
			alpha * ( fp.y + sin ( alpha * fp.y) ) ) );
		newfp.y = fp.y + h * sin( fp.x + sin(
			alpha * ( fp.x + sin ( alpha * fp.x) ) ) );
	}
	else
		{
		newfp.x =  fp.y ;
		newfp.y =  fp.x ;
		for (i=0; i < flavor; i++)
		{
			newfp.x = fp.y + sin( alpha * newfp.x );
			newfp.y = fp.x + sin( alpha * newfp.y );
		}
		newfp.x = fp.x - h * sin( newfp.x );
		newfp.y = fp.y + h * sin( newfp.y );
	}
	return newfp;
}

void	checkkeyboard() {
	int keyword,temp,i,j;
	double center, width;
	int clearflag = 0;
	if ( kbhit() ) {
		keyword = getch();
		if (keyword == 0)
		keyword = 256 + getch();
		switch (keyword) {
			case 'a':
				alpha -= 0.5;
				clearflag = 1;
				break;
			case 'A' :
				alpha += 0.5;
				clearflag = 1;
				break;
			case 'b':
				beta -= 0.2;
				if (beta < 0.2)
					beta = 0.2;
				clearflag = 1;
				break;
			case 'B' :
				beta += 0.2;
				clearflag = 1;
				break;
			case 'c':
				clearflag = 1;
				step = 0;
				break;
			case 'C':
				clearflag = 1;
				lox = -20; hix = 20; loy = -16; hiy = 16;
				step = 0;
				break;
			case 'd' :
				h /= 2;
				break;
			case 'D' :
				h *= 2;
				break;
			case 'e':
				if (showflag)
					break;
				if ( startgdriver == EGA || startgdriver == VGA )
				{
					if (gdriver ==  CGA)
					{
						gdriver = startgdriver;
						gmode = startgmode;
						startgraphics();
						textflag = 0;
					}
					else
					{
						gdriver = CGA;
						gmode = CGAC0;
						startgraphics();
						textflag = 0;
					}
				}
				break;
			case 'E':
				if (showflag)
					break;
				if ( startgdriver == VGA )
				{
					if (gdriver ==  EGA)
					{
						gdriver = startgdriver;
						gmode = startgmode;
						startgraphics();
						textflag = 0;
					}
					else
					{
						gdriver = EGA;
						gmode = EGAHI;
						startgraphics();
						textflag = 0;
					}
				}
				break;
			case 'f' :
				flavor = 0;
				clearflag = 1;
				break;
			case 'F' :
				flavor = 1;
				beta = 2;
				clearflag = 1;
				break;
			case 'g' :
				flavor = 2;
				clearflag = 1;
				break;
			case 'G' :
				if (flavor < 2)
					flavor = 2;
				else
					flavor++;
				if (flavor > 16)
					flavor = 16;
				clearflag = 1;
				break;
			case 'h':
				if (!showflag)
				{
					restorecrtmode();
					textflag = 1;
				}
				else
					clrscr();
				inform("vine.hlp");
				clearflag = 1;
				break;
			case 'H':
				if (!showflag)
				{
					restorecrtmode();
					textflag = 1;
				}
				else
					clrscr();
				inform("vine.inf");
				inform("vine.hlp");
				clearflag = 1;
				break;
			case 'i' :
				lox /= 2;
				loy /= 2;
				hix /= 2;
				hiy /= 2;
				clearflag = 1;
				break;
			case 'I' :
				lox *= 2;
				loy *= 2;
				hix *= 2;
				hiy *= 2;
				clearflag = 1;
				break;
			case 'l':
				if (! showflag)
				{
					showflag = 1;
					restorecrtmode();
					textflag = 1;
				}
				showparams();
				clearflag = 1;
				setgraphmode(gmode);
				textflag = 0;
				break;
			case 'm':
				center = randomx();
				width = (hix - lox) / 2;
				lox = center - width;
                                hix = center + width;
				center = randomy();
				width = (hiy - loy) / 2;
				loy = center - width;
				hiy = center + width;
				clearflag = 1;
				break;
			case 'n':
				if (maxiteration > 8)
					maxiteration /= 2;
				break;
			case 'N':
				if (maxiteration < 0x4000)
					maxiteration *= 2;
				break;
			case 'p':
				randomizepalette();
				break;
			case 'P':
				spinflag ^= 1;
				break;
			case 'r':
				restorecrtmode();
				textflag = 1;
				readparams( pickaname() );
				clearflag = 1;
				break;
			case 's':
				flavor = -1;
				step = 0;
				clearflag = 1;
				break;
			case 't':
				steptype++;
				if (steptype == 1)
					resolution = 10;
				if ( steptype > 4 )
					steptype = 0;
				step = 0;
				clearflag = 1;
				break;
			case 'u':
				if (resolution > 10)
					resolution -= 10;
				break;
			case 'U':
				if (resolution < maxy)
					resolution += 10;
				break;
			case 27 :	/* ESC */
			case 'Q':
			case 'q':
				exitflag = 1;
				break;
			case 13 : 	/* ENTER */
				showflag = 0;
				break;
			case 'w':
				restorecrtmode();
				textflag = 1;
				writeparams( pickaname() );
				clearflag = 1;
				break;
		}
		if (clearflag && !showflag)
		{
			if (!textflag)
				cleardevice();
			else
				setgraphmode(gmode);
			step = 0;
			screenstep = 0;
			eraseflag = 0;
			fullflag = 0;
		}
	}
}

void inform(char * filename)
{
	FILE *fp;
	char c;
	int col = 0, row = 0;
	int returnflag;
	int newline = 0;

	clrscr();
	if ( (fp = fopen(filename, "r") ) == NULL )
	{
		printf("Can't find file %s.  Press any key to continue.\n",
			filename);
		while ( !kbhit() );
		getch();
		return;
	}
	while ( (c = getc(fp) ) != EOF )
	{
		if (newline)
		{
			if (c != ' ')
				putchar(c);
			newline = 0;
		}
		else
			putchar(c);
		col++;
		if ( c == '\n' )
		{
			col = 0;
			row++;
		}
		if ( (col > 70) && c == ' ')
		{
			putchar('\n');
			col = 0;
			row++;
			newline = 1;
		}
		if (row == 22 )
		{
			printf("\n-----------------------------------------\n");
			printf("       Press any key for more.");
			while( !kbhit() );
			getch();
			row = 0;
			col = 0;
			clrscr();
		}
	}
	printf("\n----------------------------------------\n");
	printf("       Press any key to continue.");
	while( !kbhit() );
	getch();
	fclose(fp);
}

void trail(void)
{
	struct fpair nextfp, oldfp, newfp;
	int color, iteration=0;

	nextfp = oldfp = startfp();
	color = random(maxcolor) + 1;
	while( !kbhit() && iteration < maxiteration )
	{
		newfp = move(oldfp);
		show(pixel(newfp), color);
		oldfp = newfp;
		iteration++;
	}
        if (fullflag)
	{
		eraseflag = 1;
		oldfp = memory[step];
	}
	memory[step] = nextfp;
	if (spinflag && !(step & 31))
		spinpalette();
	step++;
	if (step == traillength)
	{
		fullflag = 1;
		step = 0;
	}
	if (eraseflag)
	{
        	iteration = 0;
		while( !kbhit() && iteration < maxiteration )
		{
			newfp = move(oldfp);
			show(pixel(newfp), 0);
			oldfp = newfp;
			iteration++;
		}
	}
}

struct fpair startfp(void)
{
	struct fpair fp;
	static double horizontal, xstart, xend, ystart, yend, slope, freq, amp;

	switch (steptype)
	{
		case 0:
			fp = randompoint();
			break;
		case 1:
			fp = grid();
			screenstep++;
			if (screenstep >= resolution * resolution)
			{
				screenstep = 0;
				resolution += 20;
				if (gdriver != CGA)
					randomizepalette();
				if (resolution > maxx)
				{
					cleardevice();
					step = 0;
					eraseflag = 0;
					fullflag = 0;
					resolution = 10;
					alpha += 0.5;
				}
			}
			break;
		case 2:
			fp.x = lox + screenstep * (hix - lox) / maxx;
			fp.y = loy + screenstep * (hiy - loy) / maxx;
			screenstep++;
			if (screenstep == maxx - 1)
			{
				screenstep = 0;
				alpha += 0.5;
				if (alpha > 5)
						alpha = -5;
				cleardevice();
				eraseflag = 0;
				fullflag = 0;
				step = 0;
			}
			break;
		case 3:
			if (!screenstep)
			{
				horizontal = random(2);
				if (horizontal)
				{
					ystart = randomy();
					yend = randomy();
					slope =  (yend - ystart) / (hix - lox);
				}
				else
				{
					xstart = randomx();
					xend = randomx();
					slope =  (xend - xstart) / (hiy - loy);
				}
				randomizepalette();
			}
			if (horizontal)
			{
				fp.x = lox + screenstep * (hix - lox) / maxx;
				fp.y = ystart + screenstep * slope *
					(hiy - loy) / maxx;
			}
			else
			{
				fp.y = loy + screenstep * (hiy - loy) / maxy;
				fp.x = xstart + screenstep * slope *
					(hix - lox) / maxy;
			}
			screenstep++;
			if (horizontal && screenstep == maxx ||
				!horizontal && screenstep == maxy)
				screenstep = 0;
			break;
		case 4:
			if (!screenstep)
			{
				horizontal = random(2);
				freq = ( 2 + random(7) ) * 2 * PI / (hix - lox);
				if (horizontal)
				{
					ystart = randomy();
					yend = randomy();
					slope =  (yend - ystart) / (hix - lox);
					amp = ( (10 +
						random( MIN( MIN( ystart, maxy - ystart ),
						MIN( yend, maxy - yend) ) ) ) *
						(hiy - loy) ) / maxy ;
				}
				else
				{
					xstart = randomx();
					xend = randomx();
					slope =  (xend - xstart) / (hiy - loy);
					amp = ( ( 10 +
						random( MIN( MIN( xstart, maxx - xstart ),
						MIN( xend, maxx - xend) ) ) ) *
						(hix - lox) ) / maxx;
				}
			}
			if (horizontal)
			{
				fp.x = lox + screenstep * (hix - lox) / maxx;
				fp.y = ystart +
					(screenstep * slope * (hiy - loy) / maxx) +
					amp * sin( freq * (fp.x - lox) ) ;
			}
			else
			{
				fp.y = loy + screenstep * (hiy - loy) / maxy;
				fp.x = xstart + 
					screenstep * slope * (hix - lox) / maxy +
					amp * sin( freq * (fp.y - loy) ) ;
			}
			screenstep++;
			if (horizontal && screenstep == maxx ||
				!horizontal && screenstep == maxy)
				screenstep = 0;
			break;
	}
	return fp;
}

double saw(double x)
{
	x /= (2.0 * PI);
	x -= (int) x;
	x *= 4;
	if ( -4 <= x && x < -3 )
		return (4 + x);
	if ( -3 <= x && x < -1 )
		return (-2 - x);
	if ( -1 <= x && x < 1)
		return x;
	if ( 1 <= x && x < 3)
		return (2 - x);
	else
		return (-4 + x);
}

void showparams()
{
	while (showflag)
	{
		clrscr();
		printf("Use the suggested keys to change parameter values.\n");
		printf("To go to graphics, press Enter.\n\n");
		printf("a,A.  alpha is %f\n",alpha);
		printf("b,B.  beta is %f\n",beta);
		printf("f,F,g,G,s.  Flavor is %d\n",flavor);
		printf("i,I,m.  xwindow is %f to %f, ywindow is %f to %f\n",lox,hix,loy,hiy);
		printf("d,D.  ds is %f\n",h);
		printf("n,N.  iterations per point is %d\n",maxiteration);
		printf("t. Steptype is %d\n", steptype);
		while (!kbhit());
		checkkeyboard();
	}
}

void writeparams(char * filename)
{
	FILE *fp;

//	if ( ( creat(filename,S_IREAD|S_IWRITE) ) == -1 )
	if ( (fp = fopen(filename, "wt") ) == NULL )
	{
		printf("File creation error.  Press any key to continue.\n");
		while ( !kbhit() );
		getch();
		return;
	}
	fprintf(fp, "%lf %lf %d %lf %lf %lf %lf %lf %d %d", alpha, beta,
		 flavor, lox, hix, loy, hiy, h, maxiteration, steptype);
	fclose(fp);
}

void readparams(char * filename)
{
	FILE *fp;

	if ( (fp = fopen(filename, "rt") ) == NULL )
	{
		printf("Can't find file %s.  Press any key to continue.\n",
			filename);
		while ( !kbhit() );
		getch();
		return;
	}
	fseek(fp, 0L, SEEK_SET);
	fscanf(fp, "%lf %lf %d %lf %lf %lf %lf %lf %d %d",
 &alpha, &beta, &flavor, &lox, &hix, &loy, &hiy, &h, &maxiteration, &steptype);
	fclose(fp);
}

char * pickaname()
{
	static char * filename;

	restorecrtmode();
	textflag = 1;
	printf("Give the name of the parameter file you wish write or read.\n");
	printf("Specify the name as up to eight characters, followed by a period,\n");
	printf("followed by the three letters prm.  Like: myvine.prm\n");
	printf("Press Enter when you are done.\n");
	scanf("%s", filename);
	return filename;
}