/*
      (C) Copyright 1990-91 by Autodesk, Inc.

******************************************************************************
*                                                                            *
* The information contained herein is confidential, proprietary to Autodesk, *
* Inc., and considered a trade secret as defined in section 499C of the      *
* penal code of the State of California.  Use of this information by anyone  *
* other than authorized employees of Autodesk, Inc. is granted only under a  *
* written non-disclosure agreement, expressly prescribing the scope and      *
* manner of such use.                                                        *
*                                                                            *
******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "grtypes.h"
#include "linalg.hpp"
//
// CyVector3 methods
// 
CyVector3::CyVector3(Real x, Real y, Real z)	// constructor with initialization
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
}

CyVector3::CyVector3()						// construct zero vector
{
	v[0] = v[1] = v[2] = 0.0;
}

CyVector3::CyVector3(CyVector3& w)				// copy initializer
{
	memcpy(v, w.v, sizeof(v));
}

CyVector3 CyVector3::operator+=(CyVector3& w)	// addition and assignment
{
	v[0] += w[0];
	v[1] += w[1];
	v[2] += w[2];
	return *this;
}

CyVector3 CyVector3::operator*=(CyVector3& w)	// cross product and assignment
{
	Real a, b, c;

	a = v[Y] * w.v[Z] - v[Z] * w.v[Y];
	b = v[Z] * w.v[X] - v[X] * w.v[Z];
	c = v[X] * w.v[Y] - v[Y] * w.v[X];
	v[0] = a;
	v[1] = b;
	v[2] = c;
	return *this;
}

CyVector3 CyVector3::operator-=(CyVector3& w)	// subtraction and assignment
{
	v[0] -= w[0];
	v[1] -= w[1];
	v[2] -= w[2];
	return *this;
}

void CyVector3::Print()					// print
{
	printf("(%f %f %f) ", v[0], v[1], v[2]);
}

void CyVector3::Set(Real x,Real y,Real z)	// establish values
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
}


// unary ops
CyVector3 CyVector3::operator-()			// invert
{
	v[0] = 0.0 - v[0];
	v[1] = 0.0 - v[1];
	v[2] = 0.0 - v[2];
	return *this;
}

Real CyVector3::Mag()						// length (magnitude)
{
	return (Real)sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}

CyVector3& CyVector3::Unify()					// convert to unit length
{
	Real m = this->Mag();

	if(m < EPSI)
		return *this;
	v[0] /= m;
	v[1] /= m;
	v[2] /= m;
	return *this;
}

Real CyVector3::MagUnify()		// convert to unit length, return old length

{
	Real m = this->Mag();

	if(m < EPSI)
		return m;
	v[0] /= m;
	v[1] /= m;
	v[2] /= m;
	return m;
}


// binary ops
CyVector3 CyVector3::operator+(CyVector3& w)	// add 
{
	return CyVector3(v[0] + w.v[0], v[1] + w.v[1], v[2] + w.v[2]);
}

CyVector3 CyVector3::operator-(CyVector3& w)	// subtract
{
	return CyVector3(v[0] - w.v[0], v[1] - w.v[1], v[2] - w.v[2]);
}

CyVector3 CyVector3::operator*(CyVector3& w)	// cross product
{
	Real a, b, c;

	a = v[Y] * w.v[Z] - v[Z] * w.v[Y];
	b = v[Z] * w.v[X] - v[X] * w.v[Z];
	c = v[X] * w.v[Y] - v[Y] * w.v[X];
	return CyVector3(a, b, c);
}

CyVertex3 CyVector3::operator+(CyVertex3& w) // CyVertex3 + CyVector3
{
	return CyVertex3(v[0] + w.v[0], v[1] + w.v[1], v[2] + w.v[2]);
}

Real CyVector3::Dot(CyVector3& w)				// dot-product
{
	return v[0]*w[0] + v[1]*w[1] + v[2]*w[2];
}


Real CyVector3::operator%(CyVector3& w)	// a nicer form for dot-product
{
	return v[0]*w[0] + v[1]*w[1] + v[2]*w[2];
}


// polymorphic ops (as friends due to need to commute)
CyVector3 operator*(CyVector3& w,Real r)
{
	return CyVector3(w.v[0] * r, w.v[1] * r, w.v[2] * r);
}

CyVector3 operator*(Real r,CyVector3& w)
{
	return CyVector3(w.v[0] * r, w.v[1] * r, w.v[2] * r);
}

CyVector3 operator/(CyVector3& w,Real r)
{
	return CyVector3(w.v[0] / r, w.v[1] / r, w.v[2] / r);
}
#ifndef NO_IOSTREAM
ostream& operator<<(ostream& s, CyVector3& v)
{
	s << "[" << v.v[0] << ", " << v.v[1] << ", " << v.v[2] << "]";
	return s;
}
#endif

// packetizers

/// void CyVector3::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyVector3 data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the amount of buffer taken
/// up by the encoding.
/// 
void CyVector3::EncodeToPkt(uchar* buf, int& index)
{
    int i;
    Real r;

    for(i=0;i<3;i++) {
	r = v[i];
	buf[index++] = ((uchar *) &r)[0];
	buf[index++] = ((uchar *) &r)[1];
	buf[index++] = ((uchar *) &r)[2];
	buf[index++] = ((uchar *) &r)[3];
	buf[index++] = ((uchar *) &r)[4];
	buf[index++] = ((uchar *) &r)[5];
	buf[index++] = ((uchar *) &r)[6];
	buf[index++] = ((uchar *) &r)[7];
    }
}

/// void CyVector3::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode data from a buffer <buf> of unsigned char into the CyVector3 internal
/// data, starting at <index>.  The index argument is incremented by the amount
/// of buffer taken up by the encoding.  All internal data present in the
/// CyVector3 before the operation is lost.
/// 
void CyVector3::DecodeFromPkt(uchar* buf, int& index)
{
    int i;
    Real r;
    uchar *addr_r = (uchar *) &r;

    for(i=0;i<3;i++) {
	addr_r[0] = buf[index++];
	addr_r[1] = buf[index++];
	addr_r[2] = buf[index++];
	addr_r[3] = buf[index++];
	addr_r[4] = buf[index++];
	addr_r[5] = buf[index++];
	addr_r[6] = buf[index++];
	addr_r[7] = buf[index++];
	v[i] = r;
    }
}

/// int CyVector3::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyVector3.  This will vary
/// with the compiler and machine architecture.
/// 
int CyVector3::BytesNeededToEncode()
{
    return 24;  // FIX
}

//
// Vertex class
//
CyVertex3::CyVertex3(Real x, Real y, Real z)	// constructor with initialization
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
}

CyVertex3::CyVertex3()						// construct zero vector
{
	v[0] = v[1] = v[2] = 0.0;
}

void CyVertex3::Print()					// print
{
	printf("(%f %f %f) ", v[0], v[1], v[2]);
}

CyVertex3::CyVertex3(CyVertex3& w)				// copy initializer
{
	memcpy(v, w.v, sizeof(v));
}

CyVertex3 CyVertex3::operator+=(CyVector3& w)	// addition and assignment
{
	v[0] += w[0];
	v[1] += w[1];
	v[2] += w[2];
	return *this;
}

void CyVertex3::Set(Real x,Real y,Real z)	// establish values
{
	v[0] = x;
	v[1] = y;
	v[2] = z;
}


// binary ops
CyVector3 CyVertex3::operator-(CyVertex3& w)	// subtract
{
	return CyVector3(v[0] - w.v[0], v[1] - w.v[1], v[2] - w.v[2]);
}

CyVertex3 CyVertex3::operator+(CyVector3& w) // CyVertex3 + CyVector3
{
	return CyVertex3(v[0] + w.v[0], v[1] + w.v[1], v[2] + w.v[2]);
}

CyVertex3 CyVertex3::operator-(CyVector3& w) // CyVertex3 - CyVector3
{
	return CyVertex3(v[0] - w.v[0], v[1] - w.v[1], v[2] - w.v[2]);
}

CyVertex3 CyVertex3::operator+(CyVertex3& w)	// add 
{
	return CyVertex3(v[0] + w.v[0], v[1] + w.v[1], v[2] + w.v[2]);
}

Real CyVertex3::Dist2(CyVertex3& w)			// square of dist
{
	Real a = v[0] - w.v[0];
	Real b = v[1] - w.v[1];
	Real c = v[2] - w.v[2];
	return (a*a + b*b + c*c);
}

// polymorphic ops (as friends due to need to commute)
CyVertex3 operator*(CyVertex3& w,Real r)
{
	return CyVertex3(w.v[0] * r, w.v[1] * r, w.v[2] * r);
}

CyVertex3 operator*(Real r,CyVertex3& w)
{
	return CyVertex3(w.v[0] * r, w.v[1] * r, w.v[2] * r);
}

#ifndef NO_IOSTREAM
ostream& operator<<(ostream& s, CyVertex3& v)
{
	s << "(" << v.v[0] << ", " << v.v[1] << ", " << v.v[2] << ")";
	return s;
}
#endif

// packetizers

/// void CyVertex3::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyVertex3 data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the amount of buffer taken
/// up by the encoding.
///   
void CyVertex3::EncodeToPkt(uchar* buf, int& index)
{
    int i;
    Real r;

    for(i=0;i<3;i++) {
	r = v[i];
	buf[index++] = ((uchar *) &r)[0];
	buf[index++] = ((uchar *) &r)[1];
	buf[index++] = ((uchar *) &r)[2];
	buf[index++] = ((uchar *) &r)[3];
	buf[index++] = ((uchar *) &r)[4];
	buf[index++] = ((uchar *) &r)[5];
	buf[index++] = ((uchar *) &r)[6];
	buf[index++] = ((uchar *) &r)[7];
    }
}

/// void CyVertex3::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode data from a buffer <buf> of unsigned char into the CyVertex3 internal
/// data, starting at <index>.  The index argument is incremented by the amount
/// of buffer taken up by the encoding.  All internal data present in the 
/// CyVertex3 before the operation is lost.
/// 
void CyVertex3::DecodeFromPkt(uchar* buf, int& index)
{
    int i;
    Real r;
    uchar *addr_r = (uchar *) &r;

    for(i=0;i<3;i++) {
	addr_r[0] = buf[index++];
	addr_r[1] = buf[index++];
	addr_r[2] = buf[index++];
	addr_r[3] = buf[index++];
	addr_r[4] = buf[index++];
	addr_r[5] = buf[index++];
	addr_r[6] = buf[index++];
	addr_r[7] = buf[index++];
	v[i] = r;
    }
}

/// int CyVertex3::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyVertex3.  This will vary
/// with the compiler and machine architecture.
/// 
int CyVertex3::BytesNeededToEncode()
{
    return 24;  // FIX
}

/*
	CyMatrix methods:

	Matrices are stored as 3 x 4 arrays, with the rightmost
	column reflecting translation. An associated field, mtype,
	is used to characterize a matrix as an identity, translation
	only, rotation only, or general.

	The table "mtypes" defines transitions from a given matrix
	type when composed with a second type. Note that the static
	initialization below counts on the order of the enums in
	the .hpp file. (id, trans, rot, general)
*/

static const int MTX_ID = 0, MTX_TRANS = 1, MTX_ROT = 2, MTX_GEN = 3;
static mtypes[4][4] =
{
	{MTX_ID, MTX_TRANS, MTX_ROT, MTX_GEN},
	{MTX_TRANS, MTX_TRANS, MTX_GEN, MTX_GEN},
	{MTX_ROT, MTX_GEN, MTX_ROT, MTX_GEN},
	{MTX_GEN, MTX_GEN, MTX_GEN, MTX_GEN}
};

CyMatrix::CyMatrix()					// constructor
{
	Ident();
}

CyMatrix::CyMatrix(CyMatrix& orig)			// copy initializer
{
	memcpy(m, orig.m, sizeof(m));
	mtype = orig.mtype;
}

void CyMatrix::SetQuat( class CyQuaternion& quat )
{
	CyQuaternion q = quat;
	Real vsqd[3];

	q.Normalize(); // 4 multiplies, 4 divides, 3 additions 
	vsqd[X] = q.v[X] * q.v[X];
	vsqd[Y] = q.v[Y] * q.v[Y];
	vsqd[Z] = q.v[Z] * q.v[Z];

	m[0][0] = 1 - 2 * ( vsqd[Y] + vsqd[Z] );
	m[0][1] = 2 * ( q.v[X] * q.v[Y] - q.qscalar * q.v[Z] );
	m[0][2] = 2 * ( q.v[X] * q.v[Z] + q.qscalar * q.v[Y] );
	m[1][0] = 2 * ( q.v[Z] * q.qscalar + q.v[X] * q.v[Y] );
	m[1][1] = 1 - 2 * ( vsqd[X] + vsqd[Z] );
	m[1][2] = 2 * ( q.v[Y] * q.v[Z] - q.qscalar * q.v[X] );
	m[2][0] = 2 * ( q.v[X] * q.v[Z] - q.qscalar * q.v[Y] );
	m[2][1] = 2 * ( q.v[X] * q.qscalar + q.v[Z] * q.v[Y] );
	m[2][2] = 1 - 2 * ( vsqd[X] + vsqd[Y] );
	mtype = mtypes[mtype][MTX_ROT];
}

// Make a CyMatrix from a Normalized CyQuaternion (polhemus generates these)
void CyMatrix::SetQuatN( class CyQuaternion& q )
{
	Real qterms[4];

	qterms[X] = q.v[X] * q.v[X];
	qterms[Y] = q.v[Y] * q.v[Y];
	qterms[Z] = q.v[Z] * q.v[Z];
	qterms[3] = q.qscalar * q.qscalar;

	m[0][0] = qterms[3] + qterms[X] - qterms[Y] - qterms[Z];
	m[0][1] = 2 * ( q.v[X] * q.v[Y] - q.qscalar * q.v[Z] );
	m[0][2] = 2 * ( q.v[X] * q.v[Z] + q.qscalar * q.v[Y] );
	m[1][0] = 2 * ( q.v[Z] * q.qscalar + q.v[X] * q.v[Y] );
	m[1][1] = qterms[3] - qterms[X] + qterms[Y] - qterms[Z];
	m[1][2] = 2 * ( q.v[Y] * q.v[Z] - q.qscalar * q.v[X] );
	m[2][0] = 2 * ( q.v[X] * q.v[Z] - q.qscalar * q.v[Y] );
	m[2][1] = 2 * ( q.v[X] * q.qscalar + q.v[Z] * q.v[Y] );
	m[2][2] = qterms[3] - qterms[X] - qterms[Y] + qterms[Z];
	mtype = MTX_ROT;
}

CyMatrix
CyMatrix::operator*=(CyMatrix& b)
{
	*this = (*this) * b;
	return *this;
}

void CyMatrix::U(Real x, Real y, Real z)
{
	m[0][0] = x;
	m[1][0] = y;
	m[2][0] = z;
	mtype = MTX_GEN;
}
void CyMatrix::V(Real x, Real y, Real z)
{
	m[0][1] = x;
	m[1][1] = y;
	m[2][1] = z;
	mtype = MTX_GEN;
}
void CyMatrix::N(Real x, Real y, Real z)
{
	m[0][2] = x;
	m[1][2] = y;
	m[2][2] = z;
	mtype = MTX_GEN;
}
void CyMatrix::T(Real x, Real y, Real z)
{
	m[0][3] = x;
	m[1][3] = y;
	m[2][3] = z;
	mtype = mtypes[mtype][MTX_TRANS];
}
void CyMatrix::T( CyVertex3 &v )
{
	m[0][3] = v[X];
	m[1][3] = v[Y];
	m[2][3] = v[Z];
	mtype = mtypes[mtype][MTX_TRANS];
}

/*
	build a matrix with origin at vertex from, negative z
	axis pointing toward to, and y axis determined by
	projection of up vector onto x-y plane.
*/
void
CyMatrix::SetFTU(CyVertex3& from, CyVertex3& to, CyVector3& up)
{
	CyVector3 x, z, u = up;
	int i;

	u.Unify();
	z = from - to;
	z.Unify();
	x = up * z;
	x.Unify();
	u = z * x;
	for(i=0; i<3; i++)
	{
		m[i][0] = x[i];
		m[i][1] = u[i];
		m[i][2] = z[i];
		m[i][3] = from[i];
	}
	mtype = MTX_GEN;
}

void CyMatrix::Print()				// print
{
	int i, j;
	static char *mstrings[] = {"ident", "trans", "rotate", "general"};

	printf("%s\n", mstrings[mtype]);
	for (i = 0; i < 3; i++)
	{
		for (j = 0; j < 4; j++)
			printf(/* NT */ "\t%+f", m[i][j]);
		printf(/* NT */ "\n");
	}
}

void CyMatrix::Print(char *s)			// print
{
	printf("%s\n", s);
	Print();
}

void CyMatrix::Ident()
{
	int i, j;
  
	for (i = 0; i < 3; i++)
		for (j = 0; j < 4; j++)
			m[i][j] = i == j ? 1.0 : 0.0;
	mtype = MTX_ID;
}

void CyMatrix::Rotx(Real theta)
{
	Real ts, tc;

	theta = theta * ANG_TO_RAD;
	ts = sin(theta);
	tc = cos(theta);
	Ident();
	m[1][1] = m[2][2] = tc;
	m[1][2] = -(m[2][1] = ts);
	mtype = MTX_ROT;
}

void CyMatrix::Roty(Real theta)
{
	Real ts, tc;

	theta = theta * ANG_TO_RAD;
	ts = sin(theta);
	tc = cos(theta);
	Ident();
	m[0][0] = m[2][2] = tc;
	m[0][2] = -(m[2][0] = ts);
	mtype = MTX_ROT;
}

void CyMatrix::Rotz(Real theta)
{
	Real ts, tc;

	theta = theta * ANG_TO_RAD;
	ts = sin(theta);
	tc = cos(theta);
	Ident();
	m[0][0] = m[1][1] = tc;
	m[0][1] = -(m[1][0] = ts);
	mtype = MTX_ROT;
}

void CyMatrix::RotVec( CyVector3 &vt, Real theta )
{
	Real r;
	
	CyMatrix doing, undoing;
		
	//First check to see if RotVec degenerates to Rotx, y, z etc.
		
	if( ( vt[ 0 ] == 0.0 ) && ( vt[ 1 ] == 0.0 ))
	{	
		if( vt[ 2 ] < 0.0 )
			Rotz( -theta );
		else
			Rotz( theta );
		return;
	}
	if( ( vt[ 1 ] == 0.0 ) && ( vt[ 2 ] == 0.0 ) )
	{
		if( vt[ 0 ] < 0.0 )
			Rotx( -theta );
		else
			Rotx( theta );
		return;
	}
	if( ( vt[ 2 ] == 0.0 ) && ( vt[ 0 ] == 0.0 ) )
	{
		if( vt[ 1 ] < 0.0 )
			Roty( theta );
		else
			Roty( -theta );
		return;
	}
	
	//Rotation around y, putting avel in the yz plane.
	// First make a projection from vt into the xz plane.
	CyVector3 temp( vt[ 0 ], 0.0, vt[ 2 ] ), z( 0.0, 0.0, 1.0 );
	
	r = ( -180/M_PI * acos( ( z % temp )/temp.Mag() ));
	
	// acos only resolves half of the arc: 0 to 180.
	// 
	if( vt[ 0 ] < 0 )
		r *= -1;
	
	doing.Roty( -r );
	temp = doing * vt; // temp becomes vt projected into the yz plane.
		
	// Now rotate around x until temp is inline with z.
		
	r = ( 180/M_PI * acos(( z % temp )/temp.Mag()));
	if( temp[ 1 ] < 0 )
		r *= -1;
	
	doing.ChainRotx( r );
	temp = doing * vt;  // Remove
	undoing = doing.Inverse();
	
	doing.ChainRotz( theta );
		
	*this = undoing * doing;
}

void CyMatrix::Translate(Real x, Real y, Real z)
{
	Ident();
	m[0][3] = x;
	m[1][3] = y;
	m[2][3] = z;
	mtype = MTX_TRANS;
}


// The Chain versions of these methods multiply the base class's
// matrix on the left by the desired kind of matrix.
// This is more efficient than creating a Rotx matrix and multiplying
// by it in all its members.

void CyMatrix::ChainRotx(Real theta)
{
	Real c, s;
	int j;
	CyMatrix t;

	t = *this;
	theta = theta * ANG_TO_RAD;
	c = cos(theta);
	s = sin(theta);
	for ( j=0; j<4; j++)
	{
		m[1][j] =  c * t.m[1][j] - s * t.m[2][j];
		m[2][j] =  s * t.m[1][j] + c * t.m[2][j];
	}
	mtype = mtypes[mtype][MTX_ROT];
}

void CyMatrix::ChainRoty(Real theta)
{
	Real c, s;
	int j;
	CyMatrix t;

	t = *this;
	theta = theta * ANG_TO_RAD;
	c = cos(theta);
	s = sin(theta);
	for ( j=0; j<4; j++)
	{
		m[0][j] =  c * t.m[0][j] - s * t.m[2][j];
		m[2][j] =  s * t.m[0][j] + c * t.m[2][j];
	}
	mtype = mtypes[mtype][MTX_ROT];
}

void CyMatrix::ChainRotz(Real theta)
{
	Real c, s;
	int j;
	CyMatrix t;

	t = *this;
	theta = theta * ANG_TO_RAD;
	c = cos(theta);
	s = sin(theta);
	for ( j=0; j<4; j++)
	{
		m[0][j] =  c * t.m[0][j] - s * t.m[1][j];
		m[1][j] =  s * t.m[0][j] + c * t.m[1][j];
	}
	mtype = mtypes[mtype][MTX_ROT];
}

void CyMatrix::ChainRotVec( CyVector3 &vt, Real theta )
{
	Real r;
	
	CyMatrix doing, undoing;
		
	//First check to see if RotVec degenerates to Rotx, y, z etc.
		
	if( ( vt[ 0 ] == 0.0 ) && ( vt[ 1 ] == 0.0 ))
	{	
		if( vt[ 2 ] < 0.0 )
			ChainRotz( -theta );
		else
			ChainRotz( theta );
		return;
	}
	if( ( vt[ 1 ] == 0.0 ) && ( vt[ 2 ] == 0.0 ) )
	{
		if( vt[ 0 ] < 0.0 )
			ChainRotx( -theta );
		else
			ChainRotx( theta );
		return;
	}
	if( ( vt[ 2 ] == 0.0 ) && ( vt[ 0 ] == 0.0 ) )
	{
		if( vt[ 1 ] < 0.0 )
			ChainRoty( -theta );
		else
			ChainRoty( theta );
		return;
	}
	
	//Rotation around y, putting avel in the yz plane.
	// First make a projection from vt into the xz plane.
	CyVector3 temp( vt[ 0 ], 0.0, vt[ 2 ] ), z( 0.0, 0.0, 1.0 );
	
	r = ( -180/M_PI * acos( ( z % temp )/temp.Mag() ));
	
	// acos only resolves half of the arc: 0 to 180.
	// 
	if( vt[ 0 ] < 0 )
		r *= -1;
	
	doing.Roty( r );
	temp = doing * vt; // temp becomes vt projected into the yz plane.
		
	// Now rotate around x until temp is inline with z.
		
	r = ( 180/M_PI * acos(( z % temp )/temp.Mag()));
	if( temp[ 1 ] < 0 )
		r *= -1;
	
	doing.ChainRotx( r );
	temp = doing * vt;  // Remove
	undoing = doing.Inverse();
	
	doing.ChainRotz( theta );
	
	*this = undoing * doing * *this;
}

void CyMatrix::ChainTranslate(Real x, Real y, Real z)
{ // We assume row 3 is always 0 0 0 1.
	m[0][3] += x;
	m[1][3] += y;
	m[2][3] += z;
	mtype = mtypes[mtype][MTX_TRANS];
}

void CyMatrix::ScaleInPlace(Real r) // Only scale the 3x3 part
{
	int i, j;

	for(i = 0; i < 3; i++)
		for(j = 0; j < 3; j++)
			m[i][j] *= r;
}


// This adjusts things at the element level for speed.  
// It leaves fourth column and row alone, sets first the three by
// three to Ident, then does ChainRotx(roll), ChainRoty(pitch),
// ChainRotz(yaw).

void CyMatrix::ScaleRollPitchYaw(Real r, Real ax, Real ay, Real az)
{
	Real cx, sx, cy, sy, cz, sz, temp1, temp2;

	ax *= ANG_TO_RAD;
	ay *= ANG_TO_RAD;
	az *= ANG_TO_RAD;
	cx = cos(ax);
	sx = sin(ax);
	cy = cos(ay);
	sy = sin(ay);
	cz = cos(az);
	sz = sin(az);
	temp1 = sy * cz;
	temp2 = sy * sz;
	m[X][X] = r *  cy * cz;
	m[Y][X] = r * (-sx * temp1 - cx * sz);
	m[Z][X] = r * (-cx * temp1 + sx * sz);
	m[X][Y] = r * cy * sz;
	m[Y][Y] = r * (-sx * temp2 + cx * cz);
	m[Z][Y] = r * (-cx * temp2 - sx * cz);
	m[X][Z] = r * sy;
	m[Y][Z] = r * sx * cy;
	m[Z][Z] = r * cx * cy;
//	m[3][0]=m[3][1]=m[3][2]=m[0][3]=m[1][3]=m[2][3]=0.0;
//	m[3][3]=1.0;  In the uses I've made in Tumble, I don't want this.
}

// See comments on ScaleRollPitchYaw above.
void CyMatrix::RollPitchYaw(Real ax, Real ay, Real az)
{
	Real cx, sx, cy, sy, cz, sz, temp1, temp2;

	ax *= ANG_TO_RAD;
	ay *= ANG_TO_RAD;
	az *= ANG_TO_RAD;
	cx = cos(ax);
	sx = sin(ax);
	cy = cos(ay);
	sy = sin(ay);
	cz = cos(az);
	sz = sin(az);
	temp1 = sy * cz;
	temp2 = sy * sz;
	m[X][X] =  cy * cz;
	m[Y][X] = -sx * temp1 - cx * sz;
	m[Z][X] = -cx * temp1 + sx * sz;
	m[X][Y] =  cy * sz;
	m[Y][Y] = -sx * temp2 + cx * cz;
	m[Z][Y] = -cx * temp2 - sx * cz;
	m[X][Z] =  sy;
	m[Y][Z] =  sx * cy;
	m[Z][Z] =  cx * cy;
//	m[3][0]=m[3][1]=m[3][2]=m[0][3]=m[1][3]=m[2][3]=0.0;
//	m[3][3]=1.0;
	mtype = MTX_ROT;
}

/*
	This inversion assumes a rigid motion matrix is supplied
 */
CyMatrix CyMatrix::Invert()
{
	short i,j;
	CyMatrix b;

	b.Ident();
	for (i = 0; i < 3; i++)
		for (j = 0; j < 3; j++)
		{
			b.m[i][j] = m[j][i];
			b.m[i][3] += -b.m[i][j] * m[j][3];
		}
	b.mtype = mtype;
	*this = b;
	return *this;
}

CyMatrix CyMatrix::Inverse()
{
	CyMatrix b = *this;
	
	return b.Invert();
}

// packetizers

/// void CyMatrix::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyMatrix data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the amount of buffer taken
/// up by the encoding.
/// 
void CyMatrix::EncodeToPkt(uchar* buf, int& index)
{
    int i, j;
    Real r;

    for(i=0;i<3;i++) {
	for (j=0;j<4;j++) {
	    r = m[i][j];
	    buf[index++] = ((uchar *) &r)[0];
	    buf[index++] = ((uchar *) &r)[1];
	    buf[index++] = ((uchar *) &r)[2];
	    buf[index++] = ((uchar *) &r)[3];
	    buf[index++] = ((uchar *) &r)[4];
	    buf[index++] = ((uchar *) &r)[5];
	    buf[index++] = ((uchar *) &r)[6];
	    buf[index++] = ((uchar *) &r)[7];
	}
    }
    buf[index++] = (uchar) mtype;
}

/// void CyMatrix::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode data from a buffer <buf> of unsigned char into the CyMatrix internal
/// data, starting at <index>.  The index argument is incremented by the amount of
/// buffer taken up by the encoding.  All internal data present in the CyMatrix
/// before the operation is lost.
/// 
void CyMatrix::DecodeFromPkt(uchar* buf, int& index)
{
    int i, j;
    Real r;
    uchar *addr_r = (uchar *) &r;

    for(i=0;i<3;i++) {
	for(j=0;j<4;j++) {
	    addr_r[0] = buf[index++];
	    addr_r[1] = buf[index++];
	    addr_r[2] = buf[index++];
	    addr_r[3] = buf[index++];
	    addr_r[4] = buf[index++];
	    addr_r[5] = buf[index++];
	    addr_r[6] = buf[index++];
	    addr_r[7] = buf[index++];
	    m[i][j] = r;
	}
    }
    mtype = (int) buf[index++];
}

/// int CyMatrix::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyMatrix.  This will vary
/// with the compiler and machine architecture.
///   
int CyMatrix::BytesNeededToEncode()
{
    return 97;  // FIX
}

// binary ops
CyVector3 CyMatrix::operator*(CyVector3& v)      // CyMatrix x vector
{
	register i;
 	Real sum[3];

	switch(mtype)
	{
		case MTX_ID:
		case MTX_TRANS:
			return v;
		default:
			for (i = 0; i < 3; i++)
				sum[i] = m[i][0] * v.v[X] +
						  m[i][1] * v.v[Y] +
						  m[i][2] * v.v[Z];
	}
	return CyVector3(sum[0], sum[1], sum[2]);
}

CyVertex3 CyMatrix::operator*(CyVertex3& v)       // CyMatrix x vertex
{
	register i;
 	Real sum[3];

	switch(mtype)
	{
		case MTX_ID:
			return v;
		case MTX_TRANS:
			sum[0] = v.v[X] + m[0][3];
			sum[1] = v.v[Y] + m[1][3];
			sum[2] = v.v[Z] + m[2][3];
			break;
		case MTX_ROT:
			for (i = 0; i < 3; i++)
				sum[i] = m[i][0] * v.v[X] +
						  m[i][1] * v.v[Y] +
						  m[i][2] * v.v[Z];
			break;
		case MTX_GEN:
			for (i = 0; i < 3; i++)
				sum[i] = m[i][0] * v.v[X] +
						  m[i][1] * v.v[Y] +
						  m[i][2] * v.v[Z] +
						  m[i][3];
		}
	return CyVertex3(sum[0], sum[1], sum[2]);
}

CyMatrix CyMatrix::operator*(CyMatrix& b)      // CyMatrix x CyMatrix
{
	CyMatrix t;
	register i, j, k;
	Real sum;

	switch(mtype)
	{
		case MTX_ID:
			return b;
		case MTX_TRANS:
			switch(b.mtype)
			{
				case MTX_ID:
					return *this;
				default:
					t = b;
					t.m[0][3] += m[0][3];
					t.m[1][3] += m[1][3];
					t.m[2][3] += m[2][3];
					t.mtype = mtypes[MTX_TRANS][b.mtype];
					return t;
			}
			break;
		default:	// not much difference in rotation and general cases
			switch(b.mtype)
			{
				case MTX_ID:
					return *this;
				case MTX_TRANS:
					t = *this;
					t.m[0][3] = m[0][0] * b.m[0][3] +
								 m[0][1] * b.m[1][3] +
								 m[0][2] * b.m[2][3] + m[0][3];
					t.m[1][3] = m[1][0] * b.m[0][3] +
								 m[1][1] * b.m[1][3] +
								 m[1][2] * b.m[2][3] + m[1][3];
					t.m[2][3] = m[2][0] * b.m[0][3] +
								 m[2][1] * b.m[1][3] +
								 m[2][2] * b.m[2][3] + m[2][3];
					t.mtype = MTX_GEN;
					return t;
				default:
					for (i = 0; i < 3; i++)
					{
						for (j = 0; j < 3; j++)
						{
							sum = 0.0;
							for (k=0; k<3; k++)
								sum += m[i][k] * b.m[k][j];
							t.m[i][j] = sum;
						}
						sum = 0.0;
						for (k=0; k<3; k++)
							sum += m[i][k] * b.m[k][3];
						t.m[i][3] = sum + m[i][3];
					}
					t.mtype = mtypes[mtype][b.mtype];
			}
	}
	return t;
}

#define SIZ 4

static Bool zerorow( Real mat[SIZ][2 * SIZ], short row );
static Bool swaprow( Real mat[SIZ][2 * SIZ], short row, short col );

/*
 *	invert a SIZ x SIZ matrix in place using gaussian elimination.
 *	return true if successful, otherwise false.
 */
static Bool gminvert(Real mat[SIZ][SIZ])
{
	Real imat[SIZ][2 * SIZ], recip;
	register short row, col;
	register short r, c;

	for (row = 0; row < SIZ; row++)
	{
		for (col = 0; col < SIZ; col++)
			imat[row][col] = mat[row][col];
		for (col = SIZ; col < 2 * SIZ; col++)
			if (row == (col - SIZ))
				imat[row][col] = 1.0;
			else
				imat[row][col] = 0.0;
	}
#ifdef DEBUG
	sa_printmat( imat );
#endif
		
	for (col = row = 0; row < SIZ; row++, col++)
	{
		if (REQ( imat[row][col], 0.0 ))
			if (!swaprow( imat, row, col ))
				return( FALSE );

		if (!REQ( imat[row][col], 1.0 ))
		{
			recip = 1.0 / imat[row][col];
			for (c = col + 1; c < 2 * SIZ; c++)
				imat[row][c] = imat[row][c] * recip;
			imat[row][col] = 1.0;
		}

		for (r = 0; r < SIZ; r++)
			if (r != row)
				if (!REQ( imat[r][col], 0.0 ))
				{
					for (c = col + 1; c < 2 * SIZ; c++)
						imat[r][c] = (-imat[r][col]) * imat[row][c] +
							imat[r][c];
					imat[r][col] = 0.0;
					if (zerorow( imat, r ))
						return( FALSE );
				}
#ifdef DEBUG
		sa_printmat( imat );
#endif
	}

	for (row = 0; row < SIZ; row++)
		for (col = SIZ; col < 2 * SIZ; col++)
			mat[row][col - SIZ] = imat[row][col];

	return( TRUE );
}

/*
 * swap the row with the first row beneath it which doesn't
 * have a zero in col.  return true if successful, otherwise false.
 */
static Bool swaprow( Real mat[SIZ][2 * SIZ], short row, short col )
{
	register short r, c;
	Real tmp[2 * SIZ];

	for (r = row + 1; r < SIZ; r++)
		if (!REQ( mat[r][col], 0.0 ))
		{
			for (c = 0; c < 2 * SIZ; c++)
				tmp[c] = mat[row][c];
			for (c = 0; c < 2 * SIZ; c++)
				mat[row][c] = mat[r][c];
			for (c = 0; c < 2 * SIZ; c++)
				mat[r][c] = tmp[c];
			return( TRUE );
		}
	return( FALSE );
}

/*
 *	return true if the row is all zeros, otherwise false
 */
static Bool zerorow( Real mat[SIZ][2 * SIZ], short row )
{
	register short col;

	for (col = 0; col < SIZ; col++)
		if (!REQ( mat[row][col], 0.0 ))
			return( FALSE );
	return( TRUE );
}

/*
	transfer 3 * 4 into square matrix, then fix up last row
*/
CyMatrix
CyGMatrix::Invert()
{
	Real mat[SIZ][SIZ];

	memcpy(mat, m, 12 * sizeof(Real));
	mat[3][0] = mat[3][1] = mat[3][2] = 0.0;
	mat[3][3] = 1.0;
	gminvert(mat);
	memcpy(m, mat, 12 * sizeof(Real));
	return *this;
}

/*
	determine two vectors that lie within the plane (i.e. both
	normal to the given norm

	given norm = (x, y, z), one is (-y, x, 0) (since dot product is 0)
	the other is computed via cross product.

	this breaks if x and y are both 0, so test, and use x and y axes in
	this special case.
*/
void
CyPlane::compute_vecs()
{
	if(REQ(norm[X], 0.0) && REQ(norm[Y], 0.0))
	{
		v1.Set(1, 0, 0);
		v2.Set(0, 1, 0);
		return;
	}
	v1.Set(-norm[Y], norm[X], 0);
	v2 = norm * v1;
}

CyPlane::CyPlane(CyVertex3& pp, CyVector3& n)
{
	p = pp;
	norm = n;
	compute_vecs();
}

CyPlane::CyPlane(CyPlane& plane)
{
    p = CyVertex3(plane.p);
    norm = CyVector3(plane.norm);
    compute_vecs();
}

void
CyPlane::Set(CyVertex3& pp, CyVector3& n)
{
	p = pp;
	norm = n;
	compute_vecs();
}

/*
	intersect ray with plane - returns TRUE if unique intersection, FALSE
	otherwise.
*/
Bool
CyPlane::RayInt(CyRay& r, CyVertex3& res)
{
	Real t;
	CyVector3 params;

	if((norm % r.v) > -EPSI) 
		return FALSE;
	CyGMatrix gm;
	CyVector3 v0(p[X] - r.p[X], p[Y] - r.p[Y], p[Z] - r.p[Z]);
	gm.U(r.v[X], r.v[Y], r.v[Z]);
	gm.V(-v1[X], -v1[Y], -v1[Z]);
	gm.N(-v2[X], -v2[Y], -v2[Z]);
	gm.Invert();
	params = gm * v0;
	t = params[X];
	if(t < 0) // intersection is on other side of base point of ray 
		return FALSE;
	/*
		res now contains the values of the parameteric
		equations of the line and plane - use the line
		equation to get the actual point of intersection.
	*/
	res = r.p + t * r.v;
	return TRUE;
}

// packetizers

/// void CyPlane::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyPlane data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the amount of buffer taken
/// up by the encoding.
/// 
void CyPlane::EncodeToPkt(uchar* buf, int& index)
{
    p.EncodeToPkt(buf, index);
    norm.EncodeToPkt(buf, index);
    v1.EncodeToPkt(buf, index);
    v2.EncodeToPkt(buf, index);
}

/// void CyPlane::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode data from a buffer <buf> of unsigned char into the CyPlane internal
/// data, starting at <index>.  The index argument is incremented by the amount
/// of buffer taken up by the encoding.  All internal data present in the CyPlane
/// before the operation is lost.
/// 
void CyPlane::DecodeFromPkt(uchar* buf, int& index)
{
    p.DecodeFromPkt(buf, index);
    norm.DecodeFromPkt(buf, index);
    v1.DecodeFromPkt(buf, index);
    v2.DecodeFromPkt(buf, index);
}

/// int CyPlane::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyPlane.  This will vary
/// with the compiler and machine architecture.
/// 
int CyPlane::BytesNeededToEncode()
{
    return (p.BytesNeededToEncode() + norm.BytesNeededToEncode()
	    + v1.BytesNeededToEncode() + v2.BytesNeededToEncode());
}

/// CyRay::CyRay(CyRay& ray)
/// 
/// Copier constructor.
/// 
CyRay::CyRay(CyRay& ray)
{
    p = CyVertex3(ray.p);
    v = CyVector3(ray.v);
}

// CyRay packetizers

/// void CyRay::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyRay data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the amount of buffer taken
/// up by the encoding.
/// 
void CyRay::EncodeToPkt(uchar* buf, int& index)
{
    p.EncodeToPkt(buf, index);
    v.EncodeToPkt(buf, index);
}

/// void CyRay::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode data from a buffer <buf> of unsigned char into the CyRay internal
/// data, starting at <index>.  The index argument is incremented by the amount
/// of buffer taken up by the encoding.  All internal data present in the CyRay
/// before the operation is lost.
/// 
void CyRay::DecodeFromPkt(uchar* buf, int& index)
{
    p.DecodeFromPkt(buf, index);
    v.DecodeFromPkt(buf, index);
}

/// int CyRay::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyRay.  This will vary
/// with the compiler and machine architecture.
/// 
int CyRay::BytesNeededToEncode()
{
    return (p.BytesNeededToEncode() + v.BytesNeededToEncode());
}

//
// CyQuaternion class
CyQuaternion::CyQuaternion( void )
{
	Set( 1, 0, 0, 0 );	// the identity quaternion
}

CyQuaternion::CyQuaternion( Real qt0, Real qt1, Real qt2, Real qt3 )
{
	Set( qt0, qt1, qt2, qt3 );
}

CyQuaternion::CyQuaternion( Real qt0, CyVector3 &vec )
{
	Set( qt0, vec );
}

CyQuaternion::CyQuaternion( CyQuaternion &quat )
{
	Set( quat.qscalar, quat.v );
}

void
CyQuaternion::Set( Real qt0, Real qt1, Real qt2, Real qt3 )
{
	qscalar = qt0;
	v.Set( qt1, qt2, qt3 );
}

void
CyQuaternion::Set( Real qt0, CyVector3 &vec )
{
	qscalar = qt0;
	v = vec;
}

CyQuaternion &
CyQuaternion::Normalize( void )
{
	Real m = sqrt( qscalar*qscalar + v[0]*v[0] + v[1]*v[1] + v[2]*v[2] );

	if ( m < EPSI )
		return *this;
	qscalar /= m;
	v[0] /= m;
	v[1] /= m;
	v[2] /= m;
	return *this;
}

CyQuaternion
CyQuaternion::Invert( void )
{
	Real m = qscalar * qscalar + (v % v);
	
	if ( m > EPSI )
	{
		v[X] /= -m;
		v[Y] /= -m;
		v[Z] /= -m;
		qscalar /= m;
	}
	return *this;
}

CyQuaternion
CyQuaternion::Inverse()
{
	CyQuaternion q = *this;
	
	return q.Invert();
}

void
CyQuaternion::Print()
{
	printf("( %f, ", qscalar );
	v.Print();
	printf(")\n");
}

void
CyQuaternion::Print(char *s)
{
	printf("%s\n", s);
	Print();
}

void
CyQuaternion::PrintMat(char *s)
{
	CyMatrix m;
	m.SetQuat( *this );
	m.Print( s );
}

// binary ops
CyQuaternion
CyQuaternion::operator+(CyQuaternion& quat)	// add 
{
	return CyQuaternion( qscalar + quat.qscalar, v + quat.v );
}

CyQuaternion
CyQuaternion::operator-(CyQuaternion& quat)	// subtract
{
	return CyQuaternion( qscalar - quat.qscalar, v - quat.v );
}

CyQuaternion
CyQuaternion::operator*(CyQuaternion& quat)	// multiplication
{
	if ( qscalar == 0.0 && quat.qscalar == 0.0 )
		return CyQuaternion( (v % quat.v) * -1.0, v * quat.v );
	else if ( v.Mag() == 0.0 )
		return CyQuaternion( qscalar * quat.qscalar, qscalar * quat.v ); 
	else if ( quat.v.Mag() == 0.0 )
		return CyQuaternion( quat.qscalar * qscalar, quat.qscalar * v ); 
	else
		return CyQuaternion((qscalar * quat.qscalar - (v % quat.v)),
			(qscalar * quat.v) + (quat.qscalar * v) + ( v * quat.v ));
}

Real
CyQuaternion::operator%(CyQuaternion& w)	// dot-product
{
	return ( qscalar * w.qscalar +
			v[X] * w.v[X] + v[Y] * w.v[Y] + v[Z] * w.v[Z] );
}

Bool
CyQuaternion::operator==(CyQuaternion &qtest)	// test for equality
{
	if ( fabs(qscalar - qtest.qscalar) > EPSI )
		return FALSE;
	if ( fabs(v[X] - qtest.v[X]) > EPSI )
		return FALSE;
	if ( fabs(v[Y] - qtest.v[Y]) > EPSI )
		return FALSE;
	if ( fabs(v[Z] - qtest.v[Z]) > EPSI )
		return FALSE;
	return TRUE;
}

void
CyQuaternion::SetMatrix( CyMatrix &mat )
{
	Real epsilon = EPSI;
	Real w, x, y, z, w4, x2;
	Real wsqd, xsqd, ysqd;

	wsqd = 1.0 / 4.0 * ( 1.0 + mat.m[0][0] + mat.m[1][1] + mat.m[2][2] );
	if ( wsqd > epsilon )
	{
		w = sqrt( wsqd );
		w4 = w * 4.0;
		x = ( mat.m[2][1] - mat.m[1][2] ) / w4;
		y = ( mat.m[0][2] - mat.m[2][0] ) / w4;
		z = ( mat.m[1][0] - mat.m[0][1] ) / w4;
	}
	else
	{
		w = 0.0;
		xsqd = -1.0 / 2.0 * ( mat.m[1][1] + mat.m[2][2] );
		if ( xsqd > epsilon )
		{
			x = sqrt( xsqd );
			x2 = x * 2.0;
			y = mat.m[1][0] / x2;
			z = mat.m[2][0] / x2;
		}
		else
		{
			x = 0.0;
			ysqd = 1.0 / 2.0 * ( 1.0 - mat.m[2][2] );
			if ( ysqd > epsilon )
			{
				y = sqrt( ysqd );
				z = mat.m[2][1] / ( y * 2.0 );
			}
			else
			{
				y = 0.0;
				z = 1.0;
			}
		}
	}
	Set( w, x, y, z );
}

CyVector3
CyQuaternion::GetVector( void )
{
	return v;
}

Real
CyQuaternion::GetReal( void )
{
	return qscalar;
}

// polymorphic ops (as friends due to need to commute)
CyQuaternion operator*( CyQuaternion& q, Real r )
{
	return CyQuaternion( q.qscalar * r, q.v * r );
}

CyQuaternion operator*( Real r, CyQuaternion& q )
{
	return ( q * r );
}

CyQuaternion operator/( CyQuaternion& q, Real r )
{
	if ( r > EPSI )
		return CyQuaternion( q * ( 1.0 / r ));
	else
		return q;
}

// packetizers

/// void CyQuaternion::EncodeToPkt(uchar* buf, int& index)
/// 
/// Encode the CyQuaternion data into a buffer <buf> of unsigned char, starting at
/// <index>.  The index argument is incremented by the number of bytes taken
/// up by the encoding equal to CyQuaternion::BytesNeededToEncode().
/// 
void CyQuaternion::EncodeToPkt(uchar* buf, int& index)
{
    v.EncodeToPkt(buf, index);
    buf[index++] = ((uchar *) &qscalar)[0];
    buf[index++] = ((uchar *) &qscalar)[1];
    buf[index++] = ((uchar *) &qscalar)[2];
    buf[index++] = ((uchar *) &qscalar)[3];
    buf[index++] = ((uchar *) &qscalar)[4];
    buf[index++] = ((uchar *) &qscalar)[5];
    buf[index++] = ((uchar *) &qscalar)[6];
    buf[index++] = ((uchar *) &qscalar)[7];
}

/// void CyQuaternion::DecodeFromPkt(uchar* buf, int& index)
/// 
/// Decode the data in the buffer <buf> of unsigned char into the CyQuaternion 
/// data, starting at <index>.  The index argument is incremented by the number 
/// of bytes taken up by the encoding equal to CyQuaternion::BytesNeededToEncode().
/// All data in the CyQuaternion prior to the operation is lost.
/// 
void CyQuaternion::DecodeFromPkt(uchar* buf, int& index)
{
    uchar *addr_qscalar = (uchar *) &qscalar;

    v.DecodeFromPkt(buf, index);
    addr_qscalar[0] = buf[index++];
    addr_qscalar[1] = buf[index++];
    addr_qscalar[2] = buf[index++];
    addr_qscalar[3] = buf[index++];
    addr_qscalar[4] = buf[index++];
    addr_qscalar[5] = buf[index++];
    addr_qscalar[6] = buf[index++];
    addr_qscalar[7] = buf[index++];
}

/// int CyQuaternion::BytesNeededToEncode()
/// 
/// Return the number of bytes needed to encode the CyQuaternion.  This will vary
/// with compiler and machine architecture.
/// 
int CyQuaternion::BytesNeededToEncode()
{
    return v.BytesNeededToEncode() + 8;
}

CyMatInterp::CyMatInterp( CyMatrix &start, CyMatrix &end )
{
	mstart = start;
	vstart = start.T();
	qstart.SetMatrix( start );
	qi = qstart;
	vi = vstart;
	mend = end;
	vend = end.T();
	qend.SetMatrix( end );
	vinterp = vend - vstart;
	qinterp = qend - qstart;
	theta = acos( qstart % qend );
	sintheta = sin( theta );
	interplinear = FALSE;
}

CyMatrix &
CyMatInterp::Interpolate( Real r )
{
	if ( interplinear )
		qi = qstart + r * qinterp;
	else
		if ( fabs( sintheta ) > EPSI )
			qi = qstart * sin((1.0 - r) * theta) / sintheta
				+ qend * sin( r * theta ) / sintheta;
	vi = vstart + r * vinterp;
	minterp.SetQuat( qi );
	minterp.T( vi );

	return minterp;
}

static char *rcsid = "$Header: /net/air/files/cyber/src/cdk/RCS/linalg.cpp,v 1.30 92/03/15 15:59:13 sjd Exp $";

/*
 * $Log:	linalg.cpp,v $
 * Revision 1.30  92/03/15  15:59:13  sjd
 * changed RotVec to compensate for changes in Roty (fixed tumbling code).
 * 
 * Revision 1.29  92/03/05  17:32:58  rhh
 * changed setftu again and fixed Roty problem
 * 
 * Revision 1.28  92/02/18  13:19:29  rhh
 * added MatInterp stuff (sjd)
 * 
 * Revision 1.27  92/01/01  22:37:58  carlt
 * added packetization methods for all classes
 * added new constructors for CyRay and CyPlane
 * to allow consistent use by Linda ops.
 * 
 * Revision 1.26  91/12/03  10:22:21  rhh
 * added Cy prefix to class names
 * 
 * Revision 1.25  91/11/20  14:46:38  carlt
 * Added CyVertex3::Print() method
 * 
 * Revision 1.24  91/11/14  16:44:33  sjd
 * added some CyQuaternion methods. Changed tests for m == 0.0 to
 * m < EPSI before dividing by m. Added CyMatrix::T( CyVertex3 & ) method.
 * 
 * Revision 1.23  91/10/24  14:38:12  sjd
 * added samt's RotVec() and ChainRotVec() methods.
 * 
 * Revision 1.22  91/10/21  16:00:07  rudy
 * Allow linear combination of CyVertex3 + r* CyVertex3 for finding
 * centroid of a flock of vertex3, weighted averages.
 * 
 * Revision 1.17  91/09/14  12:11:28  rhh
 * added support for CyRay and CyPlane
 * 
 * Revision 1.16  91/08/23  11:24:43  rhh
 * fixed bug in matrix composition (general * trans)
 * 
 * Revision 1.15  91/08/19  17:56:06  rhh
 * added mtype field for optimized matrix composition
 * 
 * Revision 1.14  91/08/12  10:18:43  rudy
 * Added ScaleInPlace, ScaleRollPitchYaw, and RollPitchYaw.  Latter two
 * are speedy ways of doing ChainRotx,ChainRoty,ChainRotz for use
 * in Tumble method of CyTumbler.
 * 
 * Revision 1.13  91/08/07  10:54:11  sjd
 * beefed up quaternion class and fixed some CyMatrix bugs.
 * 
 * Revision 1.12  91/07/29  10:04:44  rhh
 * Fixed SetFTU, and removed some of rudy's motion routines to
 * a separate class.
 * 
 * Revision 1.10  91/06/11  09:21:56  sjd
 * fixed include lines.
 * 
 * Revision 1.9  91/06/07  14:46:35  sjd
 * assign temporary result back to *this (minor oversight).
 * 
 * Revision 1.8  91/05/09  15:00:49  rhh
 * added +=, -=, *= operators
 * 
 * Revision 1.7  91/05/09  12:39:01  carlt
 * Added matrix scalar multiply
 * 
 * Revision 1.6  91/05/05  12:20:28  rhh
 * changed PI to M_PI throughout
 * 
 * Revision 1.5  91/02/28  14:06:22  rhh
 * Added Chain* versions of rotx rotx rotz translate
 * Added a special CosSinRotzyx method
 * Added, especially the Frenet method
 * Added use of % for dot product ( it IS kind of like a mod operator...)
 * (rudy)
 * 
 * Revision 1.4  91/02/20  15:17:10  rhh
 * changed matrix composition so that it took
 * advantage of the fact that matrices were
 * rigid motions, not general 4x4.
 * 
 * Revision 1.3  91/02/13  10:02:29  rhh
 * made the Unify method return *this so it could be part of expressions
 * 
 * Revision 1.2  91/02/10  11:53:40  rhh
 * added a new print method for matrices which takes a char * as arg.
 * 
 * Revision 1.1  91/01/18  10:36:37  rhh
 * Initial revision
 * 
 */
