/*
diblib.c   compiles to DIBLIB.DLL
*/

#include <windows.h>
#include "diblib.h"
#include "bug.h"

BOOL FAR PASCAL UnloadDIB(HBITMAP hBM)
{
   return DeleteObject( hBM );
}

HBITMAP FAR PASCAL LoadDIB(HWND hwnd, char *filename_in, BOOL errshow)
{
  LPBITMAPFILEHEADER lpbi;       /* long pointer to bitmap file header */
  LPBITMAPINFO       lpbm;       /* long pointer to bitmap info */
  GLOBALHANDLE       hGlobal;    /* global handle to the .BMP file */
  HBITMAP            hBitmap;    /* handle to the bitmap */
  LPSTR              hpStart;    /* points to beginning of .BMP file */
  LPSTR              hpMem;      /* points within .BMP file in memory */
  LPSTR              hpBits;     /* points to the bits in the bitmap */
  long               file_size;  /* size of the .BMP file */
  long               total_read; /* counter of bytes read */
  long               bytes_read; /* bytes read per _lread call */
  long               to_read;    /* number of bytes to read */
  HDC                hdc;        /* device context for this window */
  int                fh;  /* counters and the BMP file handle */

  /* some initialization */
  hBitmap=NULL;

  /* open the BMP file */
  fh=_lopen(filename_in, 0);

  /* make sure it opened correctly */
  if(fh<0){
    if (errshow)
       MessageBox(hwnd, "BMLib: Error opening file", (LPSTR)filename_in, MB_OK);
    return NULL;
  }

  /* find the length of the file */
  file_size=_llseek(fh, 0L, 2);
  _llseek(fh, 0L, 0);

  /* Attempt to allocate enough memory for the whole file. It will be */
  /* discarded after the bitmap is created. */
  hGlobal=GlobalAlloc(GMEM_MOVEABLE, (DWORD)(file_size+1));
  if(!hGlobal){
    _lclose(fh);
    if (errshow)
       MessageBox(hwnd, "BMLib: Get some more memory", (LPSTR)filename_in, MB_OK);
    return NULL;
  }

  /* lock the memory and read the BMP file into it */
  hpStart=GlobalLock(hGlobal);
  hpMem=hpStart;
  if(!hpMem){
    _lclose(fh);
    if (errshow)
       MessageBox(hwnd, "BMLib: Error locking memory", (LPSTR)filename_in, MB_OK);
    return FALSE;
  }

  /* Read the file in blocks of BLOCK_SIZE at a time.  This is due to the */
  /* fact that _lread cannot read more than 32768 bytes at a time. */
  total_read=0L;
  to_read=MIN((long)BLOCK_SIZE, file_size);
  while(total_read!=file_size){
    /* read from the file */
    bytes_read=_lread(fh, (LPSTR)hpMem, (WORD)to_read);

    /* calculate the total number of bytes read so far */
    total_read+=bytes_read;
    hpMem+=bytes_read;

    /* calculate the total number of bytes left to read */
    to_read=MIN((long)BLOCK_SIZE, file_size-total_read);
  }

  /* close the file */
  _lclose(fh);

  /* at this point, the .BMP file has been loaded OK into memory */
  lpbi=(LPBITMAPFILEHEADER)hpStart;

  /* lpbi is pointing to the beginning of the bitmap file loaded in memory. */
  /* A bitmap file consists of a BITMAPFILEHEADER structure followed by a   */
  /* BITMAPINFO structure then the array of bits.                           */
  lpbm=(LPBITMAPINFO)(lpbi+1);                    /* don't touch this line! */
  hpBits=(LPSTR)(lpbi);
  hpBits+=lpbi->bfOffBits;

//if (lpbm->biWidth > BODYPATCH_X | lpbm->biHeight BODYPATCH_Y)
//	MessageBox(hwnd, "*.BMP wider than 40 or higher than 32",
//		(LPSTR)filename_in, MB_OK);


  /* here goes... */
  hdc=GetDC(hwnd);
  hBitmap=CreateDIBitmap(hdc, (LPBITMAPINFOHEADER)lpbm, (LONG)CBM_INIT,
    (LPSTR)hpBits, (LPBITMAPINFO)lpbm, (WORD)DIB_RGB_COLORS);
  ReleaseDC(hwnd, hdc);

  /* unlock and free the .BMP file from memory */
  GlobalUnlock(hGlobal);
  GlobalFree(hGlobal);

  if(!hBitmap){
    if (errshow)
       MessageBox(hwnd, "BMLib: Error creating bitmap", (LPSTR)filename_in, MB_OK);
  }

  return hBitmap;
}


POINT FAR PASCAL DrawDIB( HDC hDC, HBITMAP hBitmap, short xStart, short yStart )
{
   BITMAP bm;
   HDC hDCmem;
   POINT ptsize, ptOrg;

   hDCmem = CreateCompatibleDC(hDC);
   SelectObject( hDCmem, hBitmap);
   SetMapMode( hDCmem, GetMapMode(hDC));

   GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);

   ptsize.x = bm.bmWidth;
   ptsize.y = bm.bmHeight;
   DPtoLP (hDC, &ptsize, 1);

   ptOrg.x = 0;
   ptOrg.y = 0;
   DPtoLP (hDCmem, &ptOrg, 1);

   BitBlt (hDC, xStart, yStart, ptsize.x, ptsize.y,
           hDCmem, ptOrg.x, ptOrg.y, SRCCOPY );

   DeleteDC (hDCmem);
   return( ptsize );
}

POINT FAR PASCAL StretchDIB( HDC hDC, HBITMAP hBitmap, short xStart, short yStart, double stretchfactor)
{
   BITMAP bm;
   HDC hDCmem;
   POINT ptsize, ptOrg;

   hDCmem = CreateCompatibleDC(hDC);
   SelectObject( hDCmem, hBitmap);
   SetMapMode( hDCmem, GetMapMode(hDC));

   GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);

   ptsize.x = bm.bmWidth;
   ptsize.y = bm.bmHeight;
   DPtoLP (hDC, &ptsize, 1);

   ptOrg.x = 0;
   ptOrg.y = 0;
   DPtoLP (hDCmem, &ptOrg, 1);

   if (stretchfactor==1)
      BitBlt (hDC, xStart, yStart, ptsize.x, ptsize.y,
              hDCmem, ptOrg.x, ptOrg.y, SRCCOPY );
   else
      StretchBlt (hDC, xStart, yStart, (short)(ptsize.x * stretchfactor), (short)(ptsize.y * stretchfactor),
                  hDCmem, ptOrg.x, ptOrg.y, ptsize.x, ptsize.y, SRCCOPY );

   DeleteDC (hDCmem);
   return( ptsize );
}



