/*********************************************************************
*  (C) Copyright 1991-1992 by Rudy Rucker for Autodesk, Inc.         *
**********************************************************************
* The information contained herein is confidential, proprietary      *
* to Autodesk,Inc., and considered a trade secret as defined in      *
* section 499C of the penal code of the State of California.  Use of *
* this information by anyone other than authorized employees of      *
* Autodesk, Inc.is grbuged only under a written non-disclosure       *
* agreement, expressly prescribing the scope and manner of such use. *
**********************************************************************/
// ------Compile Switches-----
#define ENZYME_COPY 1 // This applies only to endogamy and exogamy crosses,
	// and to crosses done from the gene viewer.
#define CENSUS_PERMUTE 1
//---Includes-----------------------------------

#include "bug.h"
#include "sound.h"
#include <windows.h>
#include <dos.h>
#include <math.h>
#include <malloc.h>
#pragma hdrstop
#include "random.h"
#include <mem.h> //for dnacopy

Real deathlevellookup[5] = DEATHLEVELLOOKUP;
Real sexlevellookup[5] = SEXLEVELLOOKUP;
Real mutationlevellookup[5] = MUTATIONLEVELLOOKUP;

extern struct bug *Bug[];
extern struct colony Colony[];
extern unsigned char colonycount, livebugcount,
	three_d_flag, wallbounceflag, globalmetricflag;
extern unsigned char weak_pal_index[], strong_pal_index[];
extern short BugLand_MAXX, BugLand_MAXY, BugLand_MAXZ, mesh_size;
extern short Pair[MAXSTATES][CASECOUNT];// Double arrays need the size.
extern unsigned char hitflag;
extern unsigned char hittable[MAXBUGS][MAXBUGS];
extern short bugeatvalue [MAXCOLONIES][MAXCOLONIES];
extern unsigned char soundflag, editbugid;
//---------------------------Used externally
void hatchbugs(void); //in bug.cpp 
void hatchcolonies(void); //in bug.cpp 
void DNA_to_RNA(struct bug*); //in bug.cpp and others.
void DNAmutate(struct bug*, Real); //in gene.cpp 
void DNAcross(struct bug *, struct bug *, struct bug *); //in gene.cpp 
void DNAtranspose(struct bug *); //in gene.cpp
void randomizegenes(struct bug *); // in bug.cpp
void zap_enzymes(struct bug *); // in bug.cpp
void enzymecopy(struct bug *, struct bug *); //in alter.cpp
void bugclone(struct bug *, struct bug *); //in bug.cpp
void checkmotion(struct colony *); // in bug.cpp
void census(struct colony *);
void census_and_reset(struct colony *);
void DNAbreed(struct colony *);
void hitbreed(void);
//-------------Used internally---
void near DNAcopy(struct bug *, struct bug *);
void near DNAdoublecross(struct bug *, struct bug *);
void near scale_fitness(struct colony *, Real);
//------------------Imported------
extern void installbugeatvalue(void);
extern void changestates(struct bug *, unsigned char);
extern void installdirectiontype(struct bug *);
extern void installmotiontype(struct bug *);
extern void recruit(struct bug *, struct bug *);
extern void sound(unsigned char);

//-----------------Hatch------------------------------
void hatchcolonies(void)
{
	unsigned char i;

	for (i=0; i<MAXCOLONIES; i++)
	{
		Colony[i].pop = 0;
		// Colony[i].rank gets set by putbugsincolonies

		Colony[i].breedtime = START_BREEDTIME;
		Colony[i].deathlevel = START_DEATHLEVEL;
		Colony[i].sexlevel = START_SEXLEVEL;
		Colony[i].mutationlevel = START_MUTATIONLEVEL;
		Colony[i].zaplevel = START_ZAPLEVEL;
		Colony[i].transpositionlevel = START_TRANSPOSITIONLEVEL;
		Colony[i].metazapcode = START_METAZAPCODE;

		Colony[i].exogamyflag= START_EXOGAMYFLAG;
		Colony[i].endogamyflag= START_ENDOGAMYFLAG;
		Colony[i].recruitpreyflag = START_RECRUITPREYFLAG;
		Colony[i].foodvalue =  START_FOODVALUE;
		Colony[i].poisonvalue = START_POISONVALUE;
		Colony[i].othervalue = START_OTHERVALUE;
		Colony[i].hitpreyvalue = START_HITPREYVALUE;
		Colony[i].hitradius = START_HITRADIUS;
		Colony[i].wallvalue = START_WALLVALUE;
		Colony[i].lowmotionvalue = START_LOWMOTIONVALUE;
		Colony[i].mintotalmotion = START_MINTOTALMOTION;
		Colony[i].targetentropy = START_TARGETENTROPY;
		Colony[i].goodentropyvalue = START_GOODENTROPYVALUE;
		Colony[i].breedflag =(Colony[i].breedtime>MAXBREEDTIME)?0:1;
		Colony[i].fitness = Colony[i].oldfitness = 0;
		Colony[i].total_scaled_fitness = 0.0;
		Colony[i].breed = DNAbreed;
	}
}

void hatchbugs(void)
{
	unsigned char i, j;

	for (i=0; i<MAXBUGS; i++)
	{
// These can be fed in as a struct or read from a file.
		Bug[i]->colonyid = MAXCOLONIES; // Meaning no colony yet.
		Bug[i]->oldbodyid = Bug[i]->bodyid = 0;
		//putbugsincolonies will fix these first two
		Bug[i]->bugnumid = i; // To be safe, putbugsincolonies does it.
		Bug[i]->bugbodyflag = START_BUGBODYFLAG;
		Bug[i]->radius = START_RADIUS;
		Bug[i]->lineflag = START_2DLINEFLAG;
		Bug[i]->states =  START_STATECOUNT;
		Bug[i]->directiontype = START_DIRECTIONTYPE;
		Bug[i]->mossflag = START_MOSSFLAG;
		Bug[i]->motiontype = START_MOTIONTYPE;
		// The START_MOTIONTYPE flag is put into
		// each bug with the meaning
		// 0 is turing, 1 is boid, 2 is beaver 
		// 3 is dog, 4 is wolf
		Bug[i]->basespeed = START_SPEED;
#ifdef CYBER
		Bug[i]->tan = Randomunitvector();
		Bug[i]->realspeed = 1.0;
		Bug[i]->watch_two = START_WATCH_TWO;
		Bug[i]->near_wall = START_NEAR_WALL;
		Bug[i]->cruisedistance = START_CRUISEDISTANCE;
		Bug[i]->maxspeed = START_MAXSPEED;
		
	// These six are Real
		Bug[i]->visionanglecosine = START_VISIONANGLECOSINE;
		Bug[i]->copyweight = START_COPYWEIGHT;
		Bug[i]->centerweight = START_CENTERWEIGHT;
		Bug[i]->attackweight = START_ATTACKWEIGHT;
		Bug[i]->fleeweight = START_FLEEWEIGHT;
		Bug[i]->veerweight = START_VEERWEIGHT;
		Bug[i]->acceleration = START_ACCELERATION;
#endif //CYBER



// These are always set this way.
		Bug[i]->bugid = i;
		
		Bug[i]->nearest_bug = i ? Bug[i-1] : Bug[i+1];
		Bug[i]->body_on = 0;
		Bug[i]->bugmovex = NULL;
		Bug[i]->bugmovey = NULL;
		Bug[i]->bugmovez = NULL;
//		Bug[i]->oldwritecolorindex = Bug[i]->writecolorindex =
//			strong_pal_index[Bug[i]->colonyid]; // strong trail
		Bug[i]->lowmotionflag = 1;
		Bug[i]->oldenergy = Bug[i]->energy = 0;
		Bug[i]->scaled_fitness = 0.0;
		Bug[i]->dir = Bug[i]->olddir = (unsigned char) (Random(Bug[i]->directions ));
		Bug[i]->readcolorcode = BLANK_CASE;
		Bug[i]->nearcolonycode = SELF_CASE;
		Bug[i]->oldmeetflag = Bug[i]->meetflag = 0;
		Bug[i]->oldcolor = Bug[i]->getcolor = START_BLANKCOLOR;
		Bug[i]->state = 0;
		Bug[i]->boidstate = 0;
		Bug[i]->bits_to_mutate = 0.0;
		Bug[i]->DNA_length = DNALENGTH;
		Bug[i]->boid_DNA_length = DNALENGTH;
		// We'll make the value correct in DNA_to_RNA
		// which gets called after this in bug.cpp.
		Bug[i]->dir_entropy = Bug[i]->entropy = 0.0;
		for (j=0; j<MAXDIRS; j++)
			Bug[i]->dirfreq[j] = 0;

		for (j=0; j<MAXBUGS; j++)
			Bug[i]->metric[j] = 0.0;
		for (j=0; j<MAXBUGS; j++)
			Bug[i]->metricsquared[j] = 0.0;
		for (j=0; j<MAXBUGS; j++)
			Bug[i]->to[j].Set(0.0, 0.0, 0.0);
		Bug[i]->tocentroid.Set(0.0, 0.0, 0.0);

// These could be read in instead if doing an exact world restore.
		Bug[i]->startx = Bug[i]->oldoldx = Bug[i]->oldx = Bug[i]->x =
			mesh_size * Random( BugLand_MAXX / mesh_size );
		Bug[i]->starty = Bug[i]->oldoldy = Bug[i]->oldy = Bug[i]->y =
			mesh_size * Random( BugLand_MAXY / mesh_size );
		Bug[i]->startz = Bug[i]->oldoldz = Bug[i]->oldz = Bug[i]->z =
			mesh_size * Random( BugLand_MAXZ / mesh_size);
#ifdef REAL_POS
		Bug[i]->real_dx = Bug[i]->real_dy = Bug[i]->real_dz = 0.0;
#endif //REAL_POS
		installdirectiontype(Bug[i]);
		// This sets Bug[i]->directions, bugmovex, bugmovey, bugmovez.
		installmotiontype(Bug[i]);
		//This sets Bug[i]->eat(), move(), and wrap().
		randomizegenes(Bug[i]);
	}
}


//-----------------------------DNA to RNA functions----------------------

void  DNA_to_RNA(struct bug* bug)
{
	unsigned char i, j;
	short k = 0;

	if (bug->motiontype != MOTION_BOID)
	{
#ifndef NEWDNA
	// N is CASECOUNT and M is MAXSTATES:
	// state0: case0dirg case0stateg case0writeg ...
	//	caseNdirg caseNstateg caseNwriteg
	// state1: case0dirg case0stateg case0writeg ...
	//	caseNdirg caseNstateg caseNwriteg
	// ...
	// stateM: case0dirg case0stateg case0writeg ...
	//	 caseNdirg caseNstateg caseNwriteg
	
		for (i=0; i<bug->states; i++)
		{
			for (j=0; j<CASECOUNT; j++)
			{
				bug->dirgene[ Pair[i][j] ] =
					CHOOSE(bug->DNA[k++], bug->directions);
				bug->stategene[ Pair[i][j] ] =
					CHOOSE(bug->DNA[k++], bug->states);
#ifdef MWC
				bug->writecolorgene[ Pair[i][j] ] =
					( (CHOOSE(bug->DNA[k++], 2)) ?
						strong_pal_index[bug->colonyid] :
						weak_pal_index[bug->colonyid] );
#endif //MWC
			}
#ifndef MWC
	// Let's keep the correct bugpalette colorindices in the bugs'
	// writecolorgene tables.
			bug->writecolorgene0[i] =
				( (CHOOSE(bug->DNA[k++], 2)) ?
					strong_pal_index[bug->colonyid] :
					weak_pal_index[bug->colonyid] );
			bug->writecolorgene1[i] =
				( (CHOOSE(bug->DNA[k++], 2)) ?
					strong_pal_index[bug->colonyid] :
				weak_pal_index[bug->colonyid] );
#endif //MWC
		}
#else //NEWDNA
	// N is CASECOUNT and M is MAXSTATES:
	// case0: state0dirg state0stateg state0writeg..
	//	 stateMdirg stateMstateg stateMwriteg
	// case1: state0dirg state0stateg state0writeg ...
	//	 stateMdirg stateMstateg stateMwriteg
	// ...
	// caseN: state0dirg state0caseg state0writeg...
	//	 stateMdirg stateMcaseg stateMwriteg
	
		for (j=0; j<CASECOUNT; j++)
		{
			for (i=0; i<bug->states; i++)
			{
				bug->dirgene[ Pair[i][j] ] =
					CHOOSE(bug->DNA[k++], bug->directions);
				bug->stategene[ Pair[i][j] ] =
					CHOOSE(bug->DNA[k++], bug->states);
			//assume MWC.
				bug->writecolorgene[ Pair[i][j] ] =
					( (CHOOSE(bug->DNA[k++], 2)) ?
						strong_pal_index[bug->colonyid] :
						weak_pal_index[bug->colonyid] );
			}
		}
#endif //NEWDNA
		bug->DNA_length = k;
		 // Length of the effective DNA being used is
		// (bug->states) * ( (CASECOUNT * 2) + 2)
		// Let's do mutations always in the active zone, and also
		// let's do crossovers in the active zone of the 
	}

	if (bug->motiontype > MOTION_TURMITE && bug->motiontype != MOTION_OWL) 
	//MOTION_BOID, MOTION_BEAVER, MOTION_DOG, MOTION_WOLF
	// In this case we read the flock parameters out of the
	// leading part of the DNA.  We use a byte per number.
	// There are currently 18 bytes.  Which is perfect,
	// as the DNA has 18 bytes of turmite info per state: This means that
	// an owl can use the 18 bytes of turmite info as boid info!
	// MOTION_OWL doesn't bother with boid_DNA_length as it uses
	// exactly the turmite DNA as its boid DNA.  Also it doesn't
	// need to convert the values here as it does this on the fly.
	// NOTE: WITH MWC, there are now 24 bytes of turmite info per state.
	{
		k=0;
	// 2 unsigned char (1 byte each)
		bug->watch_two = CHOOSE(bug->DNA[k++], 2);
		bug->maxspeed = MINSPEED + CHOOSE(bug->DNA[k++],
			MAX_MAXSPEED - MINSPEED + 1);
	// 2 unsigned short (1 byte each)
		bug->cruisedistance = MIN_CRUISEDISTANCE +
			(unsigned short)(CHOOSE(bug->DNA[k++],
			(MAX_CRUISEDISTANCE - MIN_CRUISEDISTANCE)) );
		bug->near_wall = MIN_NEAR_WALL +
			(unsigned short)CHOOSE(bug->DNA[k++],
			(MAX_NEAR_WALL - MIN_NEAR_WALL) );
	// 7 Real (2 bytes each)
// The visionanglecosine distribution is not smooth over the angles,
// it highly prefers 90 degrees.  The SMOOTH_VISION_ANGLE option corrects
// this.  The FORWARD_VISION makes the boids have less than 90 degree
// vision angle.  Remember that the field of vision is twice the vision
// angle.
#ifndef FORWARD_VISION
#ifndef SMOOTH_VISION_ANGLE
		bug->visionanglecosine = DREAL(bug->DNA[k++], bug->DNA[k++],
			2.0) - 1.0;
#else //SMOOTH_VISION_ANGLE
		bug->visionanglecosine = cos(MIN_VISIONANGLE +
			DREAL(bug->DNA[k++], bug->DNA[k++], M_PI - MIN_VISIONANGLE));
#endif //SMOOTH_VISION_ANGLE
#else // FORWARD_VISION
#ifndef SMOOTH_VISION_ANGLE
		bug->visionanglecosine = DREAL(bug->DNA[k++], bug->DNA[k++],
			1.0);
#else //SMOOTH_VISION_ANGLE
		bug->visionanglecosine = cos(MIN_VISIONANGLE +
			DREAL(bug->DNA[k++], bug->DNA[k++],
			(M_PI/2.0)- MIN_VISIONANGLE));
#endif //SMOOTH_VISION_ANGLE
#endif // FORWARD_VISION
		bug->copyweight = DREAL(bug->DNA[k++], bug->DNA[k++],
			MAX_WEIGHT);
		bug->centerweight = DREAL(bug->DNA[k++], bug->DNA[k++],
			MAX_WEIGHT);
		bug->attackweight = DREAL(bug->DNA[k++], bug->DNA[k++],
			MAX_WEIGHT);
		bug->fleeweight = DREAL(bug->DNA[k++], bug->DNA[k++],
			MAX_WEIGHT);
		bug->veerweight = MIN_VEERWEIGHT + DREAL(bug->DNA[k++],
			 bug->DNA[k++], (MAX_VEERWEIGHT - MIN_VEERWEIGHT) );
		bug->acceleration = MIN_ACCELERATION + DREAL(bug->DNA[k++], bug->DNA[k++],
			MAX_ACCELERATION_DIFF);
		switch (bug->motiontype)
		{
			case MOTION_BOID:
// Only the boid genes matter and this k is smaller than the turmite k.
				bug->boid_DNA_length = bug->DNA_length = k;
				break;
			case MOTION_WOLF:
				bug->boid_DNA_length = k;
				break;
			case MOTION_BEAVER:
			case MOTION_DOG:
				for (i=0; i<bug->states; i++)
				{
					bug->nearpreygene[i] = bug->DNA[k++];
					bug->nearpredgene[i] = (unsigned short)(256 +
						(unsigned short)(bug->DNA[k++]));
				}
				bug->boid_DNA_length = k;
	// Could this ever be bigger than DNA_length?
	// No, 18 + 2*states > 24*states, only if 18 > 22*states,
	// which could happen only if states were 0, which it never is.
				break;
		}
	}
}


//------------------Genetic Operators--------------

void  DNAmutate(struct bug* bug, Real mr)
{	// Mutate each bit with probability mr.

	static unsigned char mask[8] = {1,2,4,8,16,32,64,128};
	short i;
	short j;
	bug->bits_to_mutate += mr * 8.0 * (Real)(bug->DNA_length);
		// Mutation rate is the probability per bit of a mutation, so
		// we multiply this times the number of bits in the DNA
		// and add to the bug's running bits_to_mutate total.
		// Keep the total as a real, so it can, if need be, work its
		// way up to 1 over several calls to DNAmutate.  Execute
		// as many bit mutations as the integral part of bits_to_mutate.
		// One might conceivably mutate the same bit twice, but we
		// will simply accept that (rare and harmless) possibility.
	while (bug->bits_to_mutate >= 1.0)
	{
		i = Random(bug->DNA_length);
		j = Random(8);
		bug->DNA[i] ^= mask[j];
		bug->bits_to_mutate -= 1.0;
	}
}


void near DNAdoublecross(struct bug *p, struct bug *q)
{

	short i, breakpoint, dnalength;
	unsigned char breakbit, breakmask = 0xFF, temp;

// If p and q have differing numbers of states, we take a
// breakpoint less than the minimum, and cross up to the
// maximum.

	dnalength = MAX(p->DNA_length, q->DNA_length);
	breakpoint = Random(MIN(p->DNA_length, q->DNA_length));
	breakbit = Random(8);
	breakmask >>= breakbit;
		//To break the correlation, I swap p and q half the time.
	if (Random(2))
	{
		// Let p start with p and end with q, and let
		// q start with q and end with p.
		// So if (i < breakpoint), do nothing!  At breakpoint, swap.
		temp = p->DNA[breakpoint];
		p->DNA[breakpoint] = ( p->DNA[breakpoint] & breakmask ) |
						( q->DNA[breakpoint] & (~breakmask) );
		q->DNA[breakpoint] = ( q->DNA[breakpoint] & breakmask ) |
						( temp & (~breakmask) );
		for (i= breakpoint + 1; i < dnalength; i++)
		{
				temp = q->DNA[i];
				q->DNA[i] = p->DNA[i];
				p->DNA[i] = temp;
		}
	}
	else
	{
		// Let p end with p and start with q, and let
		// q end with q and start with p.
		// So if (i > breakpoint), do nothing!  At breakpoint, swap.
		temp = p->DNA[breakpoint];
		p->DNA[breakpoint] = ( q->DNA[breakpoint] & breakmask ) |
						( p->DNA[breakpoint] & (~breakmask) );
		q->DNA[breakpoint] = ( temp & breakmask ) |
						( q->DNA[breakpoint] & (~breakmask) );
		for (i= 0; i < breakpoint; i++)
		{
				temp = q->DNA[i];
				q->DNA[i] = p->DNA[i];
				p->DNA[i] = temp;
		}
	}

	if (!(p->motiontype > MOTION_BOID && p->motiontype != MOTION_OWL &&
		q->motiontype > MOTION_BOID && q->motiontype != MOTION_OWL))
		return; // There's nothing left to do.
	//else
		//	In the case where p and q are both part turmite and 
		//	part boid, I want to do a cross again, to make sure
		//	the boid genes get shuffled.
		// If p and q have differing numbers of states, we take a
		// breakpoint less than the minimum, and cross up to the
		// maximum.
	dnalength = MAX(p->boid_DNA_length, q->boid_DNA_length);
	breakpoint = Random(MIN(p->boid_DNA_length, q->boid_DNA_length));
	breakbit = Random(8);
	breakmask >>= breakbit;
		//To break the correlation, I swap p and q half the time.
	if (Random(2))
	{
		// Let p start with p and end with q, and let
		// q start with q and end with p.
		// So if (i < breakpoint), do nothing!  At breakpoint, swap.
		temp = p->DNA[breakpoint];
		p->DNA[breakpoint] = ( p->DNA[breakpoint] & breakmask ) |
						( q->DNA[breakpoint] & (~breakmask) );
		q->DNA[breakpoint] = ( q->DNA[breakpoint] & breakmask ) |
						( temp & (~breakmask) );
		for (i= breakpoint + 1; i < dnalength; i++)
		{
				temp = q->DNA[i];
				q->DNA[i] = p->DNA[i];
				p->DNA[i] = temp;
		}
	}
	else
	{
		// Let p end with p and start with q, and let
		// q end with q and start with p.
		// So if (i > breakpoint), do nothing!  At breakpoint, swap.
		temp = p->DNA[breakpoint];
		p->DNA[breakpoint] = ( q->DNA[breakpoint] & breakmask ) |
						( p->DNA[breakpoint] & (~breakmask) );
		q->DNA[breakpoint] = ( temp & breakmask ) |
						( q->DNA[breakpoint] & (~breakmask) );
		for (i= 0; i < breakpoint; i++)
		{
				temp = q->DNA[i];
				q->DNA[i] = p->DNA[i];
				p->DNA[i] = temp;
		}
	}
}

void DNAcross(struct bug *p, struct bug *q, struct bug *r)
{ //Use this ONLY with exogamy, endogamy or Gene Viewer breed.
	// This guy calls DNA_to_RNA for the kid himself.

	short i, breakpoint;
	unsigned char parity, breakbit, breakmask = 0xFF;
#ifndef ENZYME_COPY
// If p and q have differing numbers of states, we let
// r be the average.  The adding in of the + 1 has the effect of
// rounding up.
	r->states = (p->states + q->states + 1) / 2;
	r->DNA_length = r->states * ( (CASECOUNT * 2) + 2);
		// Can confirm this in DNA_to_RNA
#else
	enzymecopy(p,r); //sets r->DNA_length as well (to same as p)
#endif //ENZYME_COPY
	

	breakpoint = Random(MIN(p->DNA_length, q->DNA_length));
	breakbit = Random(8);
	breakmask >>= breakbit;
	parity = Random(2);
	if (parity)
		for (i=0; i < r->DNA_length; i++)
		{
			if ( i < breakpoint)
				r->DNA[i] = p->DNA[i];
			else if (i == breakpoint)
				r->DNA[i] = ( p->DNA[i] & breakmask ) |
					( q->DNA[i] & (~breakmask) );
			else
				r->DNA[i] = q->DNA[i];
		}
	else
		for (i=0; i < r->DNA_length; i++)
		{
			if ( i < breakpoint)
				r->DNA[i] = q->DNA[i];
			else if (i == breakpoint)
				r->DNA[i] = ( q->DNA[i] & breakmask ) |
					( p->DNA[i] & (~breakmask) );
			else
				r->DNA[i] = p->DNA[i];
		}
	DNA_to_RNA(r);
}


// The next routine swaps two segments of DNA within the string.
// The string is treated as a loop, so that if a segment overruns
// the end, it goes to the beginning.

void DNAtranspose(struct bug *bug)
{
	short transposon_1, transposon_length, transposon_2, i, ti;
	HANDLE transposon_copy_handle_1, transposon_copy_handle_2;
	unsigned char FAR *transposon_copy_1, FAR *transposon_copy_2;

	transposon_length = Random(bug->DNA_length / MAX_TRANSPOSON_RATIO);
	if (transposon_length < MIN_TRANSPOSON_LENGTH)
		transposon_length = MIN_TRANSPOSON_LENGTH;
	transposon_copy_handle_1 = GlobalAlloc(GMEM_MOVEABLE,
		transposon_length);
	transposon_copy_1 = (unsigned char FAR *)GlobalLock(
		transposon_copy_handle_1);
	transposon_copy_handle_2 = GlobalAlloc(GMEM_MOVEABLE,
		transposon_length);
	transposon_copy_2 = (unsigned char FAR *)GlobalLock(
		transposon_copy_handle_2);
	ti = transposon_1 = Random(bug->DNA_length);
	transposon_2 = Random(bug->DNA_length);
	for (i=0; i<transposon_length; i++)
	{
		transposon_copy_1[i] = bug->DNA[ti];
		if (++ti >= bug->DNA_length)
			ti = 0;
	}
	ti = transposon_2;
	for (i=0; i<transposon_length; i++)
	{
		transposon_copy_2[i] = bug->DNA[ti];
		if (++ti >= bug->DNA_length)
			ti = 0;
	}
	ti = transposon_1;
	for (i=0; i<transposon_length; i++)
	{
		bug->DNA[ti] = transposon_copy_2[i];
		if (++ti >= bug->DNA_length)
			ti = 0;
	}
	ti = transposon_2;
	for (i=0; i<transposon_length; i++)
	{
		bug->DNA[ti] = transposon_copy_1[i];
		if (++ti >= bug->DNA_length)
			ti = 0;
	}
	GlobalUnlock(transposon_copy_handle_1);
	GlobalFree(transposon_copy_handle_1);
	GlobalUnlock(transposon_copy_handle_2);
	GlobalFree(transposon_copy_handle_2);
}

//-------------------CLONING-------------------

void enzymecopy(struct bug *abug, struct bug *bbug)
{
	bbug->directiontype = abug->directiontype;
	bbug->motiontype = abug->motiontype;
	bbug->states = abug->states;
	if (bbug->state > bbug->states)
		bbug->state = 0;
	bbug->basespeed = abug->basespeed;
	bbug->radius = abug->radius;
	bbug->lineflag = abug->lineflag;

	bbug->DNA_length = abug->DNA_length;
	installmotiontype(bbug);
	installdirectiontype(bbug); // calls DNA_to_RNA
}

void near DNAcopy(struct bug *abug, struct bug *bbug)
{
	_fmemcpy((void far *)(bbug->DNA),
		(const void far *)(abug->DNA), abug->DNA_length);
}

void bugclone(struct bug *abug, struct bug *bbug)
{
	if (abug->bugid != bbug->bugid)
	{
		DNAcopy(abug, bbug);
		enzymecopy(abug, bbug); //calls DNA_to_RNA for bbug
		bbug->state = abug->state; // To make it act just the same
	}
}

//--------------------------Change----------------

void zap_enzymes(struct bug *bug)
{
	//Randomize the params which the colony allows to drift

	unsigned short metazapcode;

	metazapcode = Colony[bug->colonyid].metazapcode;

	if ((metazapcode & MZ_TURMITE))
		randomizegenes(bug);
	if (metazapcode & MZ_STATES)
		changestates(bug, Random(MAXSTATES) + 1);// doesn't call DNA_to_RNA
	if (metazapcode & MZ_BASESPEED)
		bug->basespeed = Random(4)+1;
	if (metazapcode & MZ_RADIUS)
		bug->radius = Random(4)?0:Random(3)+1;
	if (metazapcode & MZ_LINES)
		bug->lineflag =  Random(2);
	if (metazapcode & MZ_MOTION)
	{
		bug->motiontype = Random(MOTION_COUNT);
		installmotiontype(bug); // Doesn't call DNA_to_RNA
	}
	if (metazapcode & MZ_DIRECTIONS)
	{
		bug->directiontype = three_d_flag?Random(2):Random(5);
		installdirectiontype(bug); // doesn't calls DNA_to_RNA
	}
}

void randomizegenes(struct bug *bug)
{
	short i, k;

//#define RANDOMIZE_ALL
#ifdef RANDOMIZE_ALL
// Go ahead and randomize ALL the DNA, not just the bug->DNA_length
// part that is active.
	for (k=0; k<DNALENGTH; k++ )
#else // NOT RANDOMIZE_ALL
	for (k=0; k<bug->DNA_length; k++)
#endif //RANDOMIZE_ALL
	{
		bug->DNA[k] = Randombyte();
	}
	if (wallbounceflag) //bogus
// Ensure 180 turn in WALL_CASE.  128/255 is 0.5 so this means a
// half turn.  The gene for the dir in state i in WALL_CASE comes out
// as indicated given that we have this gene setup:
// N is CASECOUNT and M is MAXSTATES:
// state0: case0dirg case0stateg ... caseNdirg caseNstateg write0g write1g
// state1: case0dirg case0stateg ... caseNdirg caseNstateg write0g write1g
// ...
// stateM: case0dirg case0stateg ... caseNdirg caseNstateg write0g write1g
		for (i= 0; i<MAXSTATES; i++)
			bug->DNA[i*(CASECOUNT*2 + 2) + (WALL_CASE * 2)] = 128;
}

//--------------------Check Motion------------

void  checkmotion(struct colony *C)
{
	unsigned char i;
	struct bug *bug;

	for (i=0; i < C->pop; i++)
	{
		bug = Bug[C->rank[i]];
		if (bug->lowmotionflag)
			bug->energy += C->lowmotionvalue;
		// Clamp the energy value.
		if (bug->energy > MAX_ENERGY)
			bug->energy = MAX_ENERGY;
		if (bug->energy < MIN_ENERGY)
			bug->energy = MIN_ENERGY;
		// reset the motion
		bug->lowmotionflag = 1;
		bug->startx = bug->x;
		bug->starty = bug->y;
	}
}

//-----------------------Census-----------------

void census(struct colony *C)
{
	unsigned char i, j, temp;
	long sum = 0;
	unsigned char newrank[MAXBUGS];
	
	if (!C->pop)
		return;

#ifdef CENSUS_PERMUTE
// Permute the bugs so as to eliminate hereditary rank order.
	jolt_permute_buffer(C->pop);
	for (i=0; i < C->pop; i++)
		newrank[i] = C->rank[permute_buffer[i]];
	for (i=0; i < C->pop; i++)
		C->rank[i] = newrank[i];
#endif // CENSUS_PERMUTE

// Bubblesort the bugs.
	for (i=0; i < C->pop - 1; i++)
		for (j= i+1; j < C->pop; j++)	/* bubble sort */
			if ( Bug[C->rank[i]]->energy <
				Bug[C->rank[j]]->energy )
			{
				temp = C->rank[i];
				C->rank[i] = C->rank[j];
				C->rank[j] = temp;
			}

// Add energy into the sum variable and if not GOLDBERG reset each to 0.
	for (i=0; i < C->pop; i++)
		sum +=  Bug[C->rank[i]]-> energy;

// Average the sum for the fitness
	C->oldfitness = C->fitness;
	C->fitness = (short)(sum / (long)(C->pop));
		// Be sure not to be here if !C->pop

}

void census_and_reset(struct colony *C)
{
	unsigned char i;

	census(C);
	for (i=0; i < C->pop; i++)
		Bug[C->rank[i]]->energy = 0;
}

//-------------------Master Breed Routine-

void DNAbreed(struct colony *C)
{
	unsigned char i, j, child_count[MAXBUGS];
	Real pick, lo_val[MAXBUGS+1]; //lo_val can run one over
	

// First do fitness-proportionate reproduction.

// We require that C->pop >=1.
	if (!C->pop)
		return;

	// Initialize the child_count.
	for (i=0; i < C->pop; i++)
		child_count[i] = 0;

	scale_fitness(C, deathlevellookup[C->deathlevel]);
	// Scale so the most successful gets fmultiple children, the
	// average guys get one, and the below average guys get less.
	// in bug.h fmultiple goes from 1.25 to 1.75 to 2.0 with
	// deathlevel 1,2,3.

	// Calculate the lo_vals.
	// These values divide the region 0 to 1 up into slices
	// 0 = lo_val[0] < lo_val[1] < ... < lo_val[C->pop-1] < 1
	lo_val[0] = 0.0;
	lo_val[C->pop] = 1.0;
	if (!C->total_scaled_fitness)
		return;
	for (i=1; i < C->pop; i++)
		lo_val[i] = lo_val[i-1] +
			(Bug[C->rank[i]]->scaled_fitness / C->total_scaled_fitness);

	// Plan for C->pop clones proportionately to fitness.
	// Keep track of who will get how many clones in child_count.
	for (i=0; i < C->pop; i++)
	{
		pick = Randomreal(); // between 0.0 and 1.0
		for (j=1; j <= C->pop; j++)
		{ //find the j-1 so that lo_val[j-1] <= pick < lo_val[j]
			if ( pick < lo_val[j] )
			{
				child_count[j-1]++;
				break;
			}
		}
	}

	//If editbug dies or gets to have sex, make a noise!
	if (soundflag & SOUNDFLAG_BREED)
		for (j=0; j < C->pop; j++)
		{
			if (Bug[C->rank[j]]->bugid == editbugid)
			{
			if (child_count[j]) // try to make the noise several times 
//				for (i=0; i < child_count[j]; i++)
					sound(SEX_SOUND_INDEX);
			else
				sound(DEATH_SOUND_INDEX);
			}
		}


	// Fill all zero child_count slots with copies of guys with
	// child_count > 1.
	for (i=0; i < C->pop; i++)
	{
		if (!child_count[i])
		{
			for (j=0; j < C->pop; j++)
				if (child_count[j] > 1)
				{
#ifdef ENZYME_COPY
					bugclone(Bug[C->rank[j]], Bug[C->rank[i]]);
#else
					DNAcopy(Bug[C->rank[j]], Bug[C->rank[i]]);
#endif // ENZYME_COPY
					child_count[j]--; // It's given up one.
					child_count[i] = 1;
						//It's been dealt with.  
					break; // out of the for j loop.
				}
		}
	} // Now child_count should be 1 all the way through.

// Now do crossover, if desired.
	// Choose C->pop / 2 distinct pairs and cross them over.

	if ( C->sexlevel > NONE )
	{
		jolt_permute_buffer(C->pop);
		for (i=0; i+1 < C->pop; i += 2)
			if(Randomreal() <= sexlevellookup[C->sexlevel])
				DNAdoublecross(Bug[C->rank[permute_buffer[i]]],
					Bug[C->rank[permute_buffer[i+1]]]);
	}

//Mutate
	if (C->mutationlevel > NONE)
		for (j=0; j<C->pop; j++)
			DNAmutate(Bug[C->rank[j]], mutationlevellookup[C->mutationlevel]);


//Zap
	switch(C->zaplevel)
	{
		case LOW:
			if (Randomreal() <= 0.5)
				zap_enzymes( Bug[ C->rank[C->pop - 1] ] );
			break;
		case MEDIUM:
			zap_enzymes( Bug[ C->rank[C->pop - 1] ] );
			break;
		case HIGH:
			zap_enzymes( Bug[ C->rank[C->pop - 1] ] );
			if (C->pop > 1)
				zap_enzymes( Bug[ C->rank[C->pop - 2] ] );
			break;
		// Don't do anything if zaplevel is NONE
	}
	

//Transpose
	switch(C->transpositionlevel)
	{
		case LOW:
			if (Randomreal() <= 0.33)
				DNAtranspose(Bug[C->rank[0]]);
			break;
		case MEDIUM:
			if (Randomreal() <= 0.66)
				DNAtranspose(Bug[C->rank[0]]);
			break;
		case HIGH:
			DNAtranspose(Bug[C->rank[0]]);
			break;
		// Don't do anything in case NONE
	}

//reset the energy and do DNA_to_RNA all around.
	for (i=0; i < C->pop; i++)
	{
		Bug[C->rank[i]]->energy = 0;
		DNA_to_RNA(Bug[C->rank[i]]); //Try and just do it here.
	}
}

void near scale_fitness(struct colony *C, Real fmultiple)
{
// See Dan Goldberg GENETIC ALGORITHMS, Addison Wesley '89., p. 79.
//  Fitness multiple should be about 1.2 to 2.0 for pops 50 to 100.
// for small populations he suggests.  The idea is an average
// cybug should get one child and the best one should get 
// fmultiple children.  scale is u -> u' = au + b.
//----The graph should look like this, but umin' can't be negative.
//	umax' = fmultiple * uavg'   /
//						 /
//	uavg'			    /
//					  /
//	umin'			/
//
//					umin uavg umax
//	intercetp b = 
//	0 on the x axis
/* MOre precisely, in the first case, umax' = umax * a + b =
 umax * (fmultiple - 1.0) * uavg /(umax - uavg) +
(uavg *(umax - fmultiple * uavg)/(umax - uavg)) =
(fmultiple * uavg * umax - uavg * umax + uavg * umax - fmultiple * uavg^2)/
(umax - uavg) = 
(fmultiple * uavg * umax - fmultipl * uavg^2)/(umax - uavg) =
fmultiple * uavg (umax - uavg) / (umax - uavg) = fmultiple * uavg.
Also uavg' = uavg * a + b = 
 uavg * (fmultiple - 1.0) * uavg /(umax - uavg) +
(uavg *(umax - fmultiple * uavg)/(umax - uavg)) =
(fmultiple * uavg^2 - uavg^2 + uavg* umax - fmultiple * uavg^2) / 
(umax - uavg) =
uavg ( umax - uavg) / (umax - uavg) = uavg.
I don't see any reason this shouldn't work as well with MAX
and MIN_ENERGY of 10,000 instead of 1000.
*/

	unsigned char i;
	short umin = MAX_ENERGY - MIN_ENERGY, umax = 0;
	double usum = 0.0, uavg, delta, a, b;

	for (i=0; i< C->pop; i++)
	{
		Bug[C->rank[i]]->energy -= MIN_ENERGY;
		usum += Bug[C->rank[i]]->energy;
		if (Bug[C->rank[i]]->energy < umin)
			umin = Bug[C->rank[i]]->energy;
		if (Bug[C->rank[i]]->energy > umax)
			umax = Bug[C->rank[i]]->energy;
	}
	if (!C->pop)
		return;
	uavg = usum / (double)(C->pop);

	if (!(fmultiple - 1.0))
		return;
	if (umin > (((fmultiple * uavg) - umax) / (fmultiple - 1.0)))
	{	// Normal case
		delta = umax - uavg;
		if (!delta)
		{
			a = 1.0;
			b = 0.0;
		}
		else
		{
			a = (fmultiple - 1.0) * uavg / delta;
			b = uavg * (umax - (fmultiple * uavg)) / delta;
		}
	}
	else
	{ // scale as much as possible
		delta = uavg - umin;
		if (!delta)
		{
			a = 1.0;
			b = 0.0;
		}
		else
		{
			a = uavg / delta;
			b = -umin * uavg / delta;
		}
	}

	C->total_scaled_fitness = 0.0; // reset
	for (i=0; i< C->pop; i++)
	{
		Bug[C->rank[i]]->scaled_fitness =
			a * (double)(Bug[C->rank[i]]->energy) + b;
		C->total_scaled_fitness += Bug[C->rank[i]]->scaled_fitness;
	}
}

void hitbreed(void)
{
	unsigned char i, j, colonyi, colonyj;

	for (i=0;i<livebugcount;i++)
		for(j=0;j<livebugcount;j++)
			if (hittable[i][j])
			{
				colonyi = Bug[i]->colonyid;
				colonyj = Bug[j]->colonyid;
				if (colonyi == colonyj)
				{
					if (hittable[j][i]) // mutual hit
					{
						if (Colony[colonyi].endogamyflag)
						{
							DNAdoublecross(Bug[i],Bug[j]);
				//DNAcross includes DNA_to_RNA, but DNAdoublecross
				//doesn't becuase it's usually used in timed breed.
							DNA_to_RNA(Bug[i]);
							DNA_to_RNA(Bug[j]);
						}
						if (bugeatvalue[colonyi][colonyi] > 0)
						{
							Bug[i]->energy +=
								Colony[colonyi].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[i]->energy > MAX_ENERGY)
								Bug[i]->energy = MAX_ENERGY;
							if (Bug[i]->energy < MIN_ENERGY)
								Bug[i]->energy = MIN_ENERGY;
							Bug[j]->energy +=
								Colony[colonyi].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[j]->energy > MAX_ENERGY)
								Bug[j]->energy = MAX_ENERGY;
							if (Bug[j]->energy < MIN_ENERGY)
								Bug[j]->energy = MIN_ENERGY;
						}
						hittable[i][j] = 0;
						hittable[j][i] = 0;
					} // end mutual hit case
					else //not mutual
					{
						if (Colony[colonyi].endogamyflag)
							DNAcross(Bug[i],Bug[j],Bug[i]);
						if (bugeatvalue[colonyi][colonyi] > 0)
							Bug[i]->energy +=
								Colony[colonyi].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[i]->energy > MAX_ENERGY)
								Bug[i]->energy = MAX_ENERGY;
							if (Bug[i]->energy < MIN_ENERGY)
								Bug[i]->energy = MIN_ENERGY;
						hittable[i][j] = 0;
					} // end notmutual case
				} // end colonyi == colonyj case
				else // begin colonyi != colnoyj case
				{
					if (hittable[j][i]) // mutual hit
					{
						if (bugeatvalue[colonyi][colonyj] > 0)
							Bug[i]->energy +=
								Colony[colonyi].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[i]->energy > MAX_ENERGY)
								Bug[i]->energy = MAX_ENERGY;
							if (Bug[i]->energy < MIN_ENERGY)
								Bug[i]->energy = MIN_ENERGY;
						if (bugeatvalue[colonyj][colonyi] > 0)
							Bug[j]->energy +=
								Colony[colonyj].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[j]->energy > MAX_ENERGY)
								Bug[j]->energy = MAX_ENERGY;
							if (Bug[j]->energy < MIN_ENERGY)
								Bug[j]->energy = MIN_ENERGY;
						if ((Colony[colonyi].exogamyflag) &&
							(Colony[colonyj].exogamyflag))
						{
							DNAdoublecross(Bug[i],Bug[j]);
				//DNAcross includes DNA_to_RNA, but DNAdoublecross
				//doesn't becuase it's usually used in timed breed.
							DNA_to_RNA(Bug[i]);
							DNA_to_RNA(Bug[j]);
						}
						else if (Colony[colonyi].exogamyflag)
							DNAcross(Bug[i],Bug[j],Bug[i]);
						else if (Colony[colonyj].exogamyflag)
							DNAcross(Bug[i],Bug[j],Bug[j]);
						if (bugeatvalue[colonyi][colonyj] >= 0 &&
							Colony[colonyi].recruitpreyflag &&
							bugeatvalue[colonyj][colonyi] >= 0 &&
							Colony[colonyj].recruitpreyflag )
							{
								if (Random(2))
									recruit(Bug[i],Bug[j]);
								else
									recruit(Bug[j],Bug[i]);
							} //end mutual prey case
						else if ((bugeatvalue[colonyi][colonyj] >= 0) &&
							Colony[colonyi].recruitpreyflag)
							recruit(Bug[i],Bug[j]);
						else if (bugeatvalue[colonyj][colonyi] >= 0 &&
							Colony[colonyj].recruitpreyflag)
							recruit(Bug[j],Bug[i]);
						hittable[i][j] = 0;
						hittable[j][i] = 0;
					} // end mutual hit case
					else //not mutual
					{
						if (Colony[colonyi].exogamyflag)
							DNAcross(Bug[i],Bug[j],Bug[i]);
						if (bugeatvalue[colonyi][colonyj] > 0)
						{
							Bug[i]->energy +=
								Colony[colonyi].hitpreyvalue;
						// Clamp the energy values.
							if (Bug[i]->energy > MAX_ENERGY)
								Bug[i]->energy = MAX_ENERGY;
							if (Bug[i]->energy < MIN_ENERGY)
								Bug[i]->energy = MIN_ENERGY;
							if (Colony[colonyi].recruitpreyflag)
								recruit(Bug[i],Bug[j]);
						} // end prey case
						hittable[i][j] = 0;
					} // end notmutual case
				} // end colonyi != colonyi case
			}
	hitflag = 0;
}
